import React, { useState, useEffect, useCallback } from 'react';
import Modal from './Modal';
import './Modal.css';
import {  Edit } from '@mui/icons-material';
import { Button } from '@mui/material';
import Chip from '@mui/material/Chip'; 
import StyledButton from '../Buttons/StyledButton';
import AddIcon from "@mui/icons-material/Add";

const InfoModal = ({ 
  item, 
  onFileUpdated,
  onFolderUpdated,
  onClose 
}) => {
  const [showEditModal, setShowEditModal] = useState(false);
  const [showTagsModal, setShowTagsModal] = useState(false);
  const [tags, setTags] = useState([]);
  const [tagText, setTagText] = useState("");
  const [currentProp, setCurrentProp] = useState("");
  const [currentValue, setCurrentValue] = useState("");
  const [workingItem, setWorkingItem] = useState(item);
  const [flattenedObj, setFlattenedObj] = useState({});

  console.log("item at info modal:", item);

  const name = item.name ? item.name : item._name;
  const type = item._fileId ? "FileItem" : "NamedFileCollection";

  const editableProps = ['_name', 'name', '_userType', '_shortName', '_description'];

  const flattenObject = useCallback((obj, parentKey = '') => {
    let result = {};
  
    Object.keys(obj).forEach((key) => {
      const value = obj[key];
      const _key = parentKey ? `${parentKey}.${key}` : key;
      if (typeof value === 'object' && value !== null) {
        result = { ...result, ...flattenObject(value, _key) };
      } else {
        result[_key] = value;
      }
    });
  
    console.log("flattened object:", result);
    return result;
  }, []);

  useEffect(() => {
    const updateFlattenedObj = async () => {
      setFlattenedObj(flattenObject(workingItem));
    };
    updateFlattenedObj();
  }, [workingItem, flattenObject]);

  useEffect(() => {
    if (item._tags && item._tags.length > 0) {
      setTags([...item._tags]);
    }
  }, [item._tags]);

  const handleValueEdit = (event) => {
    setCurrentValue(event.target.value);
  };

  const updateWorkingItem = () => {
    setWorkingItem({ ...workingItem, [currentProp]: currentValue });
  };

  const editModal = () => {
    return(
      <Modal
        title={`Edit ${currentProp}`}
        primaryButtonText={"Save"}
        primaryButtonFunc={() => {
          updateWorkingItem();
          setShowEditModal(false);
        }}
        onClose={() => setShowEditModal(false)}
      >
        <input
          type='text'
          className='formInputField'
          value={currentValue}
          onChange={handleValueEdit} 
        />
      </Modal>
    );
  };

  const updateDbItem = async () => {
    if (type === 'FileItem') {
      onFileUpdated(workingItem);
    } else {
      onFolderUpdated(workingItem);
    }
    onClose();
  };

  const tagsModal = () => {
    console.log("rendering tags modal");

    const handleTextChange = (event) => {
      setTagText(event.target.value);
    };

    const removeTag = (tag) => {
      const updatedTags = tags.filter(t => t !== tag);
      setTags([...updatedTags]);
    };

    const updateTags = () => {
      setWorkingItem({ ...workingItem, _tags: [...tags] });
    };

    return(
      <Modal
        title={"Add Tags"}
        primaryButtonText={"Save"}
        primaryButtonFunc={() => {
          updateTags();
          setShowTagsModal(false);
        }}
        secondaryButtonText={"Add tag"}
        secondaryButtonFunc={() => {
          console.log("adding tag", tagText);
          setTags([...tags, tagText]);
          setTagText("");
        }}
        onClose={() => setShowTagsModal(false)}
      >
        <div style={{paddingBottom: "8px"}}>
          {tags.length > 0 && tags.map((tag) => {
            console.log("rendering tag chip:", tag);
            return (
              <Chip key={tag} label={tag} onDelete={() => removeTag(tag)} />
            );
          })}
        </div>
        <input
          type='text'
          className='formInputField'
          value={tagText}
          onChange={handleTextChange} 
        />
      </Modal>
    );
  };

  return(
    <>
      {showEditModal && editModal()}
      {showTagsModal && tagsModal()}
      <Modal
        title={`${name} ${type} data`}
        primaryButtonText={"Update"}
        primaryButtonFunc={() => {
          updateDbItem();
          onClose();
        }}      
        onClose={onClose}
      >
        <div style={{ textAlign: 'right' }}>
          <StyledButton 
            text={"Add Tags"} 
            onClick={() => {
              setShowTagsModal(true);
              console.log("setting setShowTagsModal to true");
            }}
          >
            <AddIcon style={{color: "var(--app-accent-color)"}} />
          </StyledButton>
        </div>
        <div className='infoTableDiv'>  
          <table style={{width: '100%', borderCollapse: 'collapse'}}>
            <thead className="infotablehead">
              <tr>
                <th className="infoTableTh" style={{width: '40%'}}><p>Property</p></th>
                <th className="infoTableTh" style={{width: '40%'}}><p>Value</p></th>
                <th className="infoTableTh" style={{width: '10%'}}><p>Edit</p></th>
              </tr>
            </thead>
            <tbody className='infoTable'>
              {Object.entries(flattenedObj).map(([key, value]) => {
                return(
                  <tr key={key} className="tablerow">
                    <td style={{width: '40%'}}><p>{key}</p></td>
                    <td style={{width: '40%'}}><p>{value}</p></td>
                    <td style={{width: '10%', verticalAlign: 'middle'}}>
                      {editableProps.includes(key) && 
                        <Button onClick={() => {
                          setCurrentProp(key);
                          setCurrentValue(value);
                          setShowEditModal(true);
                        }}>
                          <Edit />
                        </Button>
                      }
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div> 
      </Modal>
    </>
  );
};

export default InfoModal;
