/**
 * ****************************************************************************
 *
 * INVICARA INC CONFIDENTIAL __________________
 *
 * Copyright (C) [2012] - [2023] INVICARA INC, INVICARA Pte Ltd, INVICARA INDIA
 * PVT LTD All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Invicara Inc and its suppliers, if any. The intellectual and technical
 * concepts contained herein are proprietary to Invicara Inc and its suppliers
 * and may be covered by U.S. and Foreign Patents, patents in process, and are
 * protected by trade secret or copyright law. Dissemination of this information
 * or reproduction of this material is strictly forbidden unless prior written
 * permission is obtained from Invicara Inc.
 */

import React, { useState, useEffect } from "react";
import _ from "lodash";
import { ResponsiveLine } from "@nivo/line";
import { GenericMatButton } from "@dtplatform/platform-app-conflux/modules/IpaControls";
import {
  IafDataSource,
  IafItemSvc,
} from "@dtplatform/platform-api";
import Select from "react-select";
import common from "../helpers/common";
import { IafScriptEngine } from "@dtplatform/iaf-script-engine";
import DatetimePicker from "react-datetime-picker";
import "./Telemetry.css";
import InfoIcon from "@mui/icons-material/Info";
import { Spinner } from "./icons/Spinners";
import { useApplicationContext } from "../Hooks/appCtxHook";

const TelemetryChannel = ({ config, data }) => {
  const applicationContext = useApplicationContext();
  const [creatingChannel, setCreatingChannel] = useState(false);
  const [isSensorSelected, setIsSensorSelected] = useState();
  const [selectedYAxis, setSelectedYAxis] = useState({
    value: "Select...",
    label: "Select...",
  });
  const [relatedReading, setRelatedReading] = useState(null);
  const [graphData, setGraphData] = useState(null);
  const [loadingMsg, setLoadingMsg] = useState(null);
  const [userSelectedTime, setUserSelectedTime] = useState(new Date()); // State to store user-selected time
  const [model, setModel] = useState(applicationContext.selectedItems.selectedModel);

  useEffect(() => {
    getIotDevices();
  }, []);

  useEffect(() => {
    if (userSelectedTime !== null) {
      setGraphdata();
    }
  }, [selectedYAxis, userSelectedTime]);

  /**
   * @function getIotDevices
   * used to get data from backend using rest architecture(api)
   * @returns telCollections,telemetryConfigs, getRelatedReading
   */
  const getIotDevices = async () => {
    console.log("getting IoT devices data: ", data)
    setLoadingMsg("Getting IoT devices");
    let { project, ctx, _namespaces } = await common.getCurrentProjectInfo();
    if (!data.collectionIdIot) return;
    setLoadingMsg("Getting IoT device relatations");
    const elemCollRes = 
      await IafScriptEngine.getCollectionInComposite(
        model._id,
        { _userType: "rvt_elements" },
        { _namespaces: applicationContext.selectedItems.selectedProject._namespaces }
      );
    console.log('setting elem coll: ', elemCollRes);

    let relations = await IafItemSvc.getRelations(
      elemCollRes._itemId,
      {
        query: { 
          _relatedFromId: data.selectedElementId,
          _relatedUserType: 'ref_app_iot_collection' 
        },
      },
      ctx
    );
    console.log(relations);
    if (
      data.selectedElementId == relations._list[0]._relatedFromId ||
      data.selectedElementId == relations._list[1]._relatedFromId
    ) {
      setIsSensorSelected(true);
      // Get all orchestrators in project
      console.log("getting all datasources")
      setLoadingMsg("Getting orchestrator");
      let allDatasources = await IafDataSource.getOrchestrators();
      console.log("finding datasource")
      let IotDataSource = allDatasources._list.find(
        (item) => item._userType === "activemq_publish"
      );

      // Get all nameduseritems in project
      setLoadingMsg("Getting telemetry collection");
      console.log("getting nameduseritems")
      let namedItemsAll = await IafItemSvc.getNamedUserItems();
      console.log("finding nameduseritem")
      console.log("data.collectionIdIot", data.collectionIdIot);
      const telCollections = namedItemsAll._list.find(
        (item) => 
          item._userType === "ref_app_iot_collection" && 
          item._name === `${model._name} Telemetry Collection`
      );
      console.log("telCollections", telCollections);
      // Get Telemetry config
      console.log("getting telemetryConfigs")
      setLoadingMsg("Getting telemetry config");
      let telemetryConfigs = await IafItemSvc.getTelemetryConfigs(
        undefined,
        ctx,
        undefined
      );
      console.log("telemetryConfigs", telemetryConfigs);
      console.log("finding telemetryConfig")
      let telemetryConfigsIot = telemetryConfigs._list.find(
        (item) => item._shortName === "telconfig"
      );
      setLoadingMsg("Getting related readings");
      console.log("getting related readings")
      let getRelatedReading = await IafItemSvc.getRelatedReadingItems(
        telCollections._id,
        undefined,
        ctx,
        undefined
      );
      console.log("getRelatedReading", getRelatedReading);
      if (telCollections && telemetryConfigsIot) {
        setLoadingMsg(false);
        setCreatingChannel(true);
      }
      if (telCollections && telemetryConfigsIot && IotDataSource) {
        setLoadingMsg(false);
        setCreatingChannel(true);
      }
      if (!telemetryConfigsIot) {
        setLoadingMsg(false);
        setCreatingChannel(false);
      }

      if (!_.isEmpty(getRelatedReading?._list)) {
        console.log("getRelatedReading is not empty", getRelatedReading?._list);
        setRelatedReading(getRelatedReading);
      }

    } else {
      setIsSensorSelected(false);
    }
  };

  const getYAxisData = () => {
    const mapYAxisToValues = (yAxis) => {
      return (
        relatedReading?._list?.slice(-10).map((entry) => entry[yAxis]) || [
          0, 0, 0,
        ]
      );
    };

    switch (selectedYAxis.value) {
      case "co2":
        return mapYAxisToValues("CO2");
      case "temperature":
        return mapYAxisToValues("Temperature");
      case "humidity":
        return mapYAxisToValues("Humidity");
      case "als":
        return mapYAxisToValues("ALS");
      case "pm1.0":
        return mapYAxisToValues("PM1.0");
      case "pm2.5":
        return mapYAxisToValues("PM2.5");
      case "pm10":
        return mapYAxisToValues("PM10");
      case "voc":
        return mapYAxisToValues("VOC");
      default:
        return [];
    }
  };

  /**
   * @function setGraphdata
   * Used to set graph data for x axis and y axis
   */
  const setGraphdata = () => {
    console.log("setting graph data")
    setLoadingMsg("Setting graph data");
    const data = [
      {
        id: "data",
        data: relatedReading?._list
          ?.slice(-10)
          .map((x, index) => {
            const date = new Date(userSelectedTime);
            date.setMinutes(date.getMinutes() - index * 5); // Adjust initial time and reduce by 5 minutes for each reading
            const formattedDate = date.toLocaleString();

            return {
              x: `${date.getHours().toString().padStart(2, "0")}:${date
                .getMinutes()
                .toString()
                .padStart(2, "0")}`,
              y: getYAxisData()[index],
              [selectedYAxis.value]: getYAxisData()[index],
              z: formattedDate,
            };
          })
          .reverse(),
      },
    ];

    console.log("data", data);
    setLoadingMsg(false);
    setGraphData(data);
  };

  const handleUserTimeChange = (date) => {
    // Check if the selected date is valid
    if (date instanceof Date && !isNaN(date.getTime())) {
      // Check if the selected date is in the future
      const currentDate = new Date();
      if (date > currentDate) {
        // Invalid date (future date), show an error message
        console.error(
          "Selected date is in the future. Please choose a date in the past or present."
        );
        setLoadingMsg(
          "Selected date is in the future. Please choose a valid date in the past or present."
        );
      } else {
        // Valid date, update the state
        setLoadingMsg(false);
        setUserSelectedTime(date);
      }
    } else {
      // Invalid date, show an error message
      console.error("Invalid date selected. Please choose a valid date.");
      setLoadingMsg("Invalid date");
    }
  };

  /**
   * @function handleCreateChannel
   * Used to create telemetry collection and telemetry config
   */
  const handleCreateChannel = async () => {
    setLoadingMsg("Creating telemetry channel");
    let { project, ctx, _namespaces } = await common.getCurrentProjectInfo();

    setLoadingMsg("Finding telemetry collections");
    let namedItemsAll = await IafItemSvc.getNamedUserItems();
    let telCollections = namedItemsAll._list.find(
      (item) => item._userType === "ref_app_iot_collection"
    );

    setLoadingMsg("Checking telemetry configs");
    let telemetryConfigs = await IafItemSvc.getTelemetryConfigs(
      undefined,
      ctx,
      undefined
    );

    let telemetryConfigsIot = telemetryConfigs._list.find(
      (item) => item._shortName === "telconfig"
    );

    //Condition to check for telmetry config
    if (!telemetryConfigsIot) {
      console.log("defining Telemetry Configs")
      let telemetryConfig = {
        _namespaces: _namespaces,
        _shortName: "telconfig",
        _name: "Project A Telemetry Config",
        _userType: "telemetry_config",
        _configData: {
          _normalizationScript: {
            _userType: "telemetry_parser_script",
          },
          _bindings: [
            {
              collectionDesc: {
                _id: telCollections._id,
              },
              query: {
                _sourceId: "e3e052f9-0156-11d5-9301-0000863f27ad-00000131",
              },
            },
          ],
        },
      };

      let options = { transactional: false };
      try {
        //Create new telemetry config
        setLoadingMsg("Creating telemetry config");
        telemetryConfigsIot = await IafItemSvc.createTelemetryConfig(
          telemetryConfig,
          ctx,
          options
        );
      } catch (error) {
        console.error("An error occurred:", error);
      }
    }
    console.log("getting Telemetry Configs")
    telemetryConfigs = await IafItemSvc.getTelemetryConfigs(
      undefined,
      ctx,
      undefined
    );

    getIotDevices();
    setLoadingMsg("Telemetry channel created successfully");
    setGraphdata();
  };

  /**
   * @function handleCancelChannel
   * Used to delete telemetry config and orchestrators
   */
  const handleCancelChannel = async () => {
    setLoadingMsg("Cancelling telemetry channel");
    let { project, ctx, _namespaces } = await common.getCurrentProjectInfo();
    setLoadingMsg("Getting telemetry config");
    let telemetryConfigs = await IafItemSvc.getTelemetryConfigs(
      undefined,
      ctx,
      undefined
    );
    console.log("finding Telemetry Configs")
    let telemetryConfigsIot = telemetryConfigs._list.find(
      (item) => item._shortName === "telconfig"
    );
    if (telemetryConfigsIot) {
      try {
        // Delete telemetry config
        setLoadingMsg("Deleting telemetry config");
        console.log("deleting Telemetry Config")
        await IafItemSvc.deleteTelemetryConfig(
          telemetryConfigsIot._id,
          ctx,
          undefined
        );
      } catch (error) {
        console.error("An error occurred:", error);
      }
    }
    setLoadingMsg("Getting orchestrator");
    let allDatasources = await IafDataSource.getOrchestrators();
    console.log("finding datasources")
    let IotDataSource = allDatasources._list.find(
      (item) => item._userType === "activemq_publish"
    );
    if (IotDataSource) {
      setLoadingMsg("Deleting orchestrator");
      // Delete active mq orchestrator
      let deleteIotOrchestrator = await IafDataSource.deleteOrchestrator(
        IotDataSource.id
      );
    }
    getIotDevices();
  };

  /**
   * @function handlePublishData
   * Creates Active mq orchestrator
   */
  const handlePublishData = async () => {
    console.log("handlePublishData running")
    let { project, ctx, _namespaces } = await common.getCurrentProjectInfo();
    setLoadingMsg("Checking for MQTT orchestrator");
    let allDatasources = await IafDataSource.getOrchestrators();
    let IotDataSource = allDatasources._list.find(
      (item) => item._userType === "activemq_publish"
    );
    setLoadingMsg("Checking for IoT collection");
    let namedItemsAll = await IafItemSvc.getNamedUserItems();
    let telCollections = namedItemsAll._list.find(
      (item) => 
        item._userType === "ref_app_iot_collection"
      && 
        item._name === `${model._name} Telemetry Collection`
    );
    if (!IotDataSource) {
      // Create active mq orchestrator
      console.log("Creating MQTT orchestrator")
      setLoadingMsg("Creating MQTT orchestrator");
      let IotOrchestrator = await IafScriptEngine.addDatasource({
        _name: "ActiveMQ Publish Message",
        _description: "ActiveMQ Publish Message",
        _namespaces: _namespaces,
        _userType: "activemq_publish",
        _instant: true,
        _params: {
          tasks: [
            {
              name: "default_script_target",
              _sequenceno: 1,
              TelemetryCollectionId: telCollections._id,
              sensorId: "e3e052f9-0156-11d5-9301-0000863f27ad-00000131",
              _actualparams: {
                userType: "orchestrator",
                _scriptName: "IotSensorData",
              },
            },
            {
              name: "activemq_target",
              _sequenceno: 2,
              _actualparams: {
                activemq: {},
                topic: "VirtualTopic/" + _namespaces,
              },
            },
          ],
        },
      });
      console.log("IotOrchestrator", IotOrchestrator);
    }
    setLoadingMsg("Running MQTT orchestrator");
    let query = { _namespaces, _name: "ActiveMQ Publish Message" };
    let orchList = await IafScriptEngine.getDatasources(query, ctx);
    for (var orch in orchList) {
      if (orchList[orch]._name === "ActiveMQ Publish Message") {
        let req = {
          orchestratorId: orchList[orch].id,
        };
        let iotOrchRunResult = await IafScriptEngine.runDatasource(req, ctx);
        const readings = iotOrchRunResult._result.message.data.value.device_list;
        const relReadings = {
          _list: readings,
          total: readings.length,
          _pageSize: 10,
          _offset: 0
        };
        console.log("iotOrchRunResult relReadings: ", relReadings);
        setRelatedReading(relReadings);
        setUserSelectedTime(new Date);
        setLoadingMsg(false);
      }
    }
  };

  const handleYAxisChange = (selectedOption) => {
    //getIotDevices();
    setSelectedYAxis(selectedOption);
  };

  const yAxisOptions = [
    { value: "co2", label: "CO2" },
    { value: "temperature", label: "Temperature" },
    { value: "humidity", label: "Humidity" },
    { value: "als", label: "ALS" },
    { value: "pm1.0", label: "PM1.0" },
    { value: "pm2.5", label: "PM2.5" },
    { value: "pm10", label: "PM10" },
    { value: "voc", label: "VOC" },
  ];

  if (isSensorSelected) {
    return (
      <div>
        {creatingChannel ? (
          <div style={{ width: "500px" }}>
            <div style={{ display: "flex", flexDirection: "row" }}>
              <div
                style={{
                  paddingRight: "10px",
                }}
              >
                <GenericMatButton onClick={handleCancelChannel}>
                  Cancel Telemetry channel
                </GenericMatButton>
              </div>
              <div
                style={{
                  paddingRight: "10px",
                }}
              >
                <GenericMatButton onClick={handlePublishData}>
                  Publish Data
                </GenericMatButton>
              </div>
              <div
                className="dbm-tooltip"
                style={{
                  display: "flex",
                  flexDirection: "column",
                  justifyContent: "center",
                  transform: "scale(0.95)",
                  position: "relative",
                  zIndex: "1",
                }}
              >
                <InfoIcon
                  className="info-icon"
                  style={{
                    cursor: "pointer",
                    color: "white",
                    backgroundColor: "black",
                    borderRadius: "50%",
                  }}
                  fontSize="large"
                />
                <span
                  className="dbm-tooltiptext"
                  style={{
                    bottom: "50%",
                    right: "100%",
                    transform: "translate(61px, 66%) scale(0.80)",
                    position: "absolute",
                    width: "220px",
                    backgroundColor: "white",
                    color: "black",
                  }}
                  dangerouslySetInnerHTML={{
                    __html: `
                    Telemetry Channel support fetches a sensor's IoT data and displays the values on graph.When a user clicks <strong>CREATE TELEMETRY CHANNEL</strong>, a channel and orchestrator are created. Click <strong>PUBLISH DATA</strong> to trigger the orchestrator run and create new readings from ActiveMQ.To delete the channel and orchestrator, click <strong>CANCEL IOT CHANNEL</strong>. `,
                  }}
                ></span>
              </div>
            </div>
            {loadingMsg && (
              <>
                <h5 style={{ marginTop: "10px" }}>{loadingMsg}...</h5>
                <Spinner/>              
              </>
            )}
            <br></br>
            {graphData && (
              <div>
                <label htmlFor="timeInput">Last 10 readings upto: </label>
                <DatetimePicker
                  onChange={handleUserTimeChange}
                  value={userSelectedTime}
                  format="MM/dd/yyyy HH:mm"
                  amPm={false}
                />
                <br />
              </div>
            )}
            {graphData && (
              <div>
                <Select
                  value={selectedYAxis}
                  onChange={handleYAxisChange}
                  options={yAxisOptions}
                  styles={{
                    width: "-webkit-fill-available",
                    borderWidth: "medium",
                    control: (provided, state) => ({
                      ...provided,
                      borderColor: "var(--app-accent-color)",
                      boxShadow:
                        "0px 0px 1px var(--app-accent-color) !important",
                      "&:hover": {
                        borderColor: "var(--app-accent-color) !important", // Change this to your hover control color
                        boxShadow:
                          "0px 0px 1px var(--app-accent-color) !important",
                      },
                    }),
                    option: (provided, state) => ({
                      ...provided,
                      backgroundColor: state.isSelected
                        ? "var(--fancytree-one-color)" // Selected option background color
                        : "white", // Default option background color
                      "&:hover": {
                        backgroundColor: state.isSelected
                          ? "var(--fancytree-one-color)" // Selected option background color on hover
                          : "var(--fancytree-one-color)", // Default option background color on hover
                        color: "white",
                      },
                    }),
                  }}
                />
              </div>
            )}
            {graphData && graphData[0].data && (
              <div style={{ height: "300px" }}>
                <ResponsiveLine
                  data={graphData}
                  width={500}
                  height={300}
                  margin={{ top: 50, right: 50, bottom: 80, left: 50 }}
                  xScale={{
                    type: "point",
                  }}
                  xFormat={(value) => `${value}`}
                  axisBottom={{
                    orient: "bottom",
                    tickSize: 5,
                    tickPadding: 5,
                    tickRotation: -90,
                    legend: "Time",
                    legendOffset: 75,
                    legendPosition: "middle",
                  }}
                  axisLeft={{
                    orient: "left",
                    tickSize: 5,
                    tickPadding: 5,
                    tickRotation: 0,
                    legend: selectedYAxis.label,
                    legendOffset: -40,
                    legendPosition: "middle",
                  }}
                  enableGridX={false}
                  enableGridY={true}
                  colors={["var(--app-accent-color)"]}
                  enablePoints={false}
                  pointSize={8}
                  pointColor={{ from: "color", modifiers: [] }}
                  enableArea={false}
                  animate={true}
                  motionStiffness={90}
                  motionDamping={15}
                  useMesh={true}
                  tooltip={({ point }) => (
                    <div
                      style={{
                        background: "white",
                        padding: "9px 12px",
                        border: "1px solid #ccc",
                      }}
                    >
                      <strong>Time:</strong> {point.data.z}
                      <br />
                      <strong>{selectedYAxis.label}:</strong>{" "}
                      {point.data.yFormatted}
                    </div>
                  )}
                />
              </div>
            )}
          </div>
        ) : (
          <div>
            <GenericMatButton onClick={() => handleCreateChannel()}>
              Create Telemetry channel
            </GenericMatButton>
            {loadingMsg && (
              <>
                <h5 style={{ marginTop: "10px" }}>{loadingMsg}...</h5>
                <Spinner/>              
              </>
            )}
          </div>
        )}
      </div>
    );
  } else {
    return <h5>Selected Element is not a Telemetry device</h5>;
  }
};

export const TelemetryChannelFactory = {
  create: ({ config, data }) => {
    return <TelemetryChannel config={config} data={data} />;
  },
};
export default TelemetryChannel;
