/**
 * ****************************************************************************
 *
 * INVICARA INC CONFIDENTIAL __________________
 *
 * Copyright (C) [2012] - [2023] INVICARA INC, INVICARA Pte Ltd, INVICARA INDIA
 * PVT LTD All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Invicara Inc and its suppliers, if any. The intellectual and technical
 * concepts contained herein are proprietary to Invicara Inc and its suppliers
 * and may be covered by U.S. and Foreign Patents, patents in process, and are
 * protected by trade secret or copyright law. Dissemination of this information
 * or reproduction of this material is strictly forbidden unless prior written
 * permission is obtained from Invicara Inc.
 */

import { IafViewerDBM, IafEvmUtils } from "@dtplatform/iaf-viewer";
import _ from "lodash";
import React, { useCallback, useMemo } from "react";
import { listIncludes, listEquals } from "@dtplatform/platform-app-conflux/modules/IpaUtils";
import { extractSpacesFromEntities } from "../../components/EntityEnabledIafViewer";
import * as EntitySelectionPanel from "../entities/EntitySelectionPanel";
import * as Tourism from "./tourism/states"

// Tourism Imports
import { enterHome, eventHandler } from "./tourism/eventHandler";
import { states } from "./tourism/states";
import { getInitialContext } from "./tourism/states";
import { BASIC_GIS_CONTROL, getGisProperties, getGisOverviewProperties, getGisOverviewPropertiesForDisctrict } from "./tourism/gis/properties";
import { getUeProperties } from "./tourism/ue/properties";
import { getPhotoSphereProperties } from "./tourism/photo/properties";
import IafUtils from "../utils/IafUtils";
import CompareView from "./comparison/CompareView";
import { IafProj } from "@dtplatform/platform-api";

const modelViewerIdsList = [];
class ViewerWrapper extends React.Component {
  constructor(props) {
    super(props);

    IafUtils.platformEvmCommandModeEnabled();
    IafUtils.platformEvmPropModeEnabled();
    IafUtils.platformEvmPropMode = IafUtils.platformEvmPropMode || !IafUtils.platformEvmCommandMode;

    const enableEvmDemo = JSON.parse(ViewerWrapper.getUrlSearchParams().get("enableEvmDemo".toLowerCase()));
    this.enableComparison = JSON.parse(ViewerWrapper.getUrlSearchParams().get("enableComparison".toLowerCase()));
    console.log ('enableEvmDemo', enableEvmDemo);
    console.log ('enableComparison', this.enableComparison);

    this.state = {
      title: "IafViewer-SingleInstance",
      view2d: {
        enable: !enableEvmDemo && !this.enableComparison,
        displayMode: IafEvmUtils.EVMDisplayMode.DEFAULT
      },

      view3d: {
        enable: !enableEvmDemo,
        showToolbar: false,
        camera: undefined,
        onCameraUpdate: {
          delay: 300,
          callback: (camera) => {
            const currentCamera = this.state.view3d.camera;

            // const isCameraChanged = Object.keys(camera).some(
            //   key => camera[key] !== currentCamera?.[key]
            // );
            const isCameraChanged = JSON.stringify(camera) !== JSON.stringify(currentCamera);

            if (isCameraChanged) {
              this.setState({
                view3d: { 
                  ...this.state.view3d ,
                  camera
                }
              });
            }
          }
        }
      },

      gis: {
        enable: !enableEvmDemo,
        showToolbar: false
      },

      // Tourism States
      arcgis: {
        enable: enableEvmDemo,
        ...getGisProperties(),
        showToolbar: true,
        displayMode: IafEvmUtils.EVMDisplayMode.DEFAULT
      },
      
      arcgisOverview: {
        enable: false,
        ...getGisOverviewPropertiesForDisctrict(),
        showToolbar: false
      },

      ue: {
        enable: false,
        ...getUeProperties(),
        showToolbar: true
      },      

      photosphere: {
        enable: false,
        ...getPhotoSphereProperties(),
        showToolbar: true
      },

      currentState: states.home, // Finite states (CITY_VIEW or DISTRICT_VIEW)
      context: getInitialContext(), // Initialize context from imported function      
      previouslyClickedModelIds: [],
      selectedElementIds: [],
      isCtrlPressed: false,
    };
    this.iafViewerDBMRef = React.createRef();
    if(this.props.enableOptimizedSelection){
      this.highlightedElementIds = _.memoize((_selectedEntities) =>
        (_selectedEntities || []).map((e) => e)
      );
    } else {
      this.highlightedElementIds = _.memoize((_selectedEntities) =>
        (_selectedEntities || []).map((e) => e.modelViewerIds[0])
      );
    }
  }
  
  static getUrlSearchParams = () => {
      let urlParams = new URLSearchParams(window.location.hash);
      urlParams = new URLSearchParams(
          Array.from(urlParams, ([key, value]) => [key.toLowerCase(), value])
      );
      return urlParams;
  }  
  //RRP:- This is moved above and behind feature flag enableOptimizedSelection.
  // highlightedElementIds = _.memoize((_selectedEntities) =>
  //   (_selectedEntities || []).map((e) => e.modelViewerIds[0])
  // );
  extractSpacesFromEntitiesMemo = _.memoize((_isolatedEntities) =>
    (extractSpacesFromEntities || [])(_isolatedEntities)
  );
  isolatedElementIds = _.memoize((_isolatedRemainingEntities) =>
    (_isolatedRemainingEntities || [])
      .map((e) => e.modelViewerIds[0])
      .filter((e) => e !== undefined)
  );
  sliceElementIds = _.memoize((_isolatedRemainingEntities) =>
    (_isolatedRemainingEntities || [])
      .map((e) => e.modelViewerIds[0])
      .filter((e) => e !== undefined)
  );
  spaceElementIds = _.memoize((_isolatedSpaces) =>
    (_isolatedSpaces || [])
      .map((e) => e.modelViewerIds[0])
      .filter((e) => e !== undefined)
  );

  resetGlassMode = async () => {
    let commands = _.get(
      this.iafViewerDBMRef,
      "current.iafviewerRef.current.commands"
    );
    if (commands && commands.setDrawMode) {
      //reset glass mode
      await commands.setDrawMode(
        false /*glassMode*/,
        false /*glassModeFromToolbar*/,
        undefined /*newDrawMode*/
      );
    }
  };
  
  /**
   * Event handler for keydown and keyup events.
   * Determines if the Ctrl or Meta key is pressed and updates the state.
   * Used to toggle the `isCtrlPressed` state for handling multi-selection.
   * @param {KeyboardEvent} event - The key event triggered by the user.
   */
  handleKey = (event) => {
     // Update the state to track whether Ctrl/Meta key is pressed.
    this.setState({ isCtrlPressed: event.ctrlKey || event.metaKey });
  };
  
  componentDidMount() {
    if(this.props.enableOptimizedSelection){
      window.addEventListener("keydown", this.handleKey);
      window.addEventListener("keyup", this.handleKey);
    }
  }

  componentWillUnmount() {
    if(this.props.enableOptimizedSelection){
      window.removeEventListener("keydown", this.handleKey);
      window.removeEventListener("keyup", this.handleKey);
    }
  }

  // --------------- Tourism Methods Begin ------------------------
  // Update the context object
  // updateContext = (newContext) => {
  //   this.setState((prevState) => ({
  //     context: { ...prevState.context, ...newContext }
  //   }));
  // };

  handleCityViewEntry = () => {
    // const entryCommands = getCityViewEntryCommands(this.state.context);
    // this.updateContext({
    //   buildingName: undefined,
    //   commandsBatch: this.state.context.commandsBatch.concat(entryCommands),
    //   sketchMode: false,
    //   districtName: undefined,
    //   districtSymbologyApplied: true,
    // });
  };

  handleCityViewExit = () => {
    // const exitCommands = getCityViewExitCommands(this.state.context);
    // this.updateContext({
    //   commandsBatch: this.state.context.commandsBatch.concat(exitCommands),
    //   districtSymbologyApplied: false,
    // });
  };

  handleBuildingSelected = (buildingName) => {
    // this.setState({ currentState: 'DISTRICT_VIEW' });
    // this.updateContext({
    //   buildingName,
    //   // commandsBatch: this.state.context.commandsBatch.concat({ type: 'ZOOM_TO_DISTRICT', district: 'SomeDistrict' })
    //   commandsBatch: this.state.context.commandsBatch.concat(zoomToDistrictOverheadAction(buildingName)),
    // });
  };

  handleDistrictSelected = (districtName) => {
    // this.setState({ currentState: 'DISTRICT_VIEW' });
    // this.updateContext({
    //   commandsBatch: this.state.context.commandsBatch.concat(
    //     zoomToDistrictAction(districtName) // Use the action here
    //   ),
    // });
  };


  handleUpdateCamera = (camera) => {
    // this.updateContext({
    //   // commandsBatch: this.state.context.commandsBatch.concat({ type: 'SET_CAMERA', camera })
    //   commandsBatch: this.state.context.commandsBatch.concat(setCameraAction(camera)),
    // });
  };

    // render() {
  //   const { currentState, context } = this.state;
  //   return (
  //     <div>
  //       <h2>Current State: {currentState}</h2>
  //       {currentState === 'CITY_VIEW' && (
  //         <button onClick={() => this.handleBuildingSelected('Building A')}>
  //           Select Building
  //         </button>
  //       )}
  //       {currentState === 'DISTRICT_VIEW' && (
  //         <div>
  //           <button onClick={() => this.handleDistrictSelected('District A')}>
  //             Select District
  //           </button>
  //           <button onClick={() => this.handleUpdateCamera({ x: 10, y: 10 })}>
  //             Update Camera
  //           </button>
  //         </div>
  //       )}
  //       <div>
  //         <h4>Command Batch:</h4>
  //         <pre>{JSON.stringify(context.commandsBatch, null, 2)}</pre>
  //       </div>
  //     </div>
  //   );
  // }  

  // --------------- Tourism Methods End ------------------------

  componentDidUpdate(prevProps, prevState, snapshot) {

    // Tourism
    {
      if (prevState.currentState !== this.state.currentState) {
        if (this.state.currentState === 'CITY_VIEW') {
          this.handleCityViewEntry();
        } else {
          this.handleCityViewExit();
        }
      }
    }

    // BIM
    if (
      this.props.isolatedEntities !== prevProps.isolatedEntities &&
      _.isEmpty(this.props.isolatedEntities)
    ) {
      this.resetGlassMode();
    }
    /* leaving it here, might be useful for debugging later on
        Object.keys(prevProps)
            .filter(key => {
                return prevProps[key] !== this.props[key];
            }).map(key => {
                console.log('changed property:',key,'from',this.props[key],'to',prevProps[key]);
            });
         */
  }

  componentDidMount() {

    // Tourism 
    {
      // this.handleCityViewEntry(); // Handle entry when component mounts
      Tourism.enterHome(this);
      // setTimeout(() => Tourism.enterModel(this), 3000);
      // this.state.arcgis.eventHandler = this.eventHandler;
    }
  }

  getModelEntities = async () => {
    let commands = _.get(
      this.iafViewerDBMRef,
      "current.iafviewerRef.current.commands"
    );
    if (commands && commands.getSelectedEntities) {
      let pkgIds = await commands.getSelectedEntities();
      if (pkgIds && pkgIds.length > 0) {
        let result = [];
        for (const pkgId of pkgIds) {
          if (isNaN(pkgId)) {
            result.push({ id: pkgId });
          } else {
            result.push({ id: parseInt(pkgId) });
          }
        }
        return result;
      } else {
        return [];
      }
    }
  };

  selectEntities = async () => {
    const modelSelectedEntities = await this.getModelEntities();
    const modelSelectedEntitiesIds = !modelSelectedEntities
      ? []
      : modelSelectedEntities.map(({ id }) => id);

    if (modelSelectedEntities && modelSelectedEntities.length > 0) {
      //sync UI => take the model as single source of truth, and sync UI
      const highlightedElementIdsOutOfSync = !listIncludes(
        _.sortBy(this.highlightedElementIds(this.props.selectedEntities)),
        _.sortBy(modelSelectedEntitiesIds)
      );
      const previouslyClickedModelIdsOutOfSync = !listIncludes(
        _.sortBy(this.state.previouslyClickedModelIds),
        _.sortBy(modelSelectedEntitiesIds)
      );
      if (
        previouslyClickedModelIdsOutOfSync ||
        highlightedElementIdsOutOfSync
      ) {
        this.props.onSelect(modelSelectedEntities);
        this.setState({ previouslyClickedModelIds: modelSelectedEntitiesIds });
      }
    } else {
      //clear UI => if we are in assets view
      const { isolatedRemainingEntities } = this.extractSpacesFromEntitiesMemo(
        this.props.isolatedEntities
      );
      if (
        this.sliceElementIds(isolatedRemainingEntities).length > 0 &&
        this.state.previouslyClickedModelIds &&
        this.state.previouslyClickedModelIds.length > 0
      ) {
        this.props.onSelect(modelSelectedEntities);
        this.setState({ previouslyClickedModelIds: modelSelectedEntitiesIds });
      }
    }
  };

  /**
   * Callback to handle changes in the selected elements (via UI interaction).
   * If the Ctrl or Command key is pressed, it toggles selection of elements: adds them if not already selected, 
   * or removes them if they are selected.
   * If the Ctrl or Command key is not pressed, it selects only the provided elements, clearing any previous selection.
   * 
   * @param {Array} elementIds - The list of element IDs to be selected or deselected.
   */
  handleOnSelectedElementChangeCallback = (elementIds) => {
    this.setState((prevState) => {
      let newSelection;
  
      if (prevState.isCtrlPressed) {
        // Toggle selection (add if not present, remove if already selected)
        newSelection = [...prevState.selectedElementIds];
        elementIds.forEach((id) => {
          if (newSelection.includes(id)) {
            newSelection = newSelection.filter((existingId) => existingId !== id); // Deselect
          } else {
            newSelection.push(id); // Select
          }
        });
      } else {
        newSelection = elementIds;
      }
      const modelSelectedEntities = newSelection.map((id) => ({ id }));
      this.props.onSelect(modelSelectedEntities);
      return { selectedElementIds: newSelection };
    });
  };

render() {
  const { isolatedSpaces, isolatedRemainingEntities } =
    this.extractSpacesFromEntitiesMemo(this.props.isolatedEntities);

  /**
   * For legacy implementations, `onClick={this.selectEntities}` is required
   * to maintain compatibility. Newer applications should use the callback mechanism
   * (OnSelectedElementChangeCallback) for better reactivity and maintainability.
   */
  const viewerProps = {
    ref: this.iafViewerDBMRef,
    ...this.props,
    _container: undefined, // Remove KeepAlive container detail to prevent unnecessary re-renders
    view2d: this.state.view2d,
    view3d: this.state.view3d,
    gis: {
      ...this.state.gis,
      token: this.props.mapboxToken,
      onIafMapReady: (map) => this.props.enableGisDrawer?.(map),
    },
    arcgis: this.state.arcgis,
    arcgisOverview: this.state.arcgisOverview,
    ue: this.state.ue,
    photosphere: this.state.photosphere,
    sliceElementIds: this.sliceElementIds(this.props.isolatedEntities),
    highlightedElementIds: this.highlightedElementIds(this.props.selectedElementIds),
    isolatedElementIds: this.isolatedElementIds(isolatedRemainingEntities),
    spaceElementIds: this.spaceElementIds(isolatedSpaces),
    selection: this.highlightedElementIds(this.props.selectedElementIds),
    OnSelectedElementChangeCallback: this.handleOnSelectedElementChangeCallback,
    allEntities: this.props.allEntities,
    workflow: this.props.workflow,
    enablePersistence: true,
    title: this.state.title
  };  

  const currentProject = IafProj.getCurrent();
  const currentModels = currentProject?._userAttributes?.currentModels || [];

  const formattedModels = currentModels.map(item => ({
    _id: item.model.model,
    _name: item.bimpk.filename.replace(/\.[^/.]+$/, ''),
    _versionId: item.model.modelVersionId,
    _namespaces: item.model?._namespaces
  }));  

  if (this.enableComparison) {
    if (formattedModels.length > 1) return (
        <div style={{ width: '100%', height: '100%' }}>
        <CompareView mode="parallel" orientation="horizontal">
          <div style={{ height: '100%', width: '100%' }}          
            onClick={!this.props.enableOptimizedSelection ? this.selectEntities : undefined}>
            <IafViewerDBM 
              {...viewerProps}
              model={formattedModels[0]}
              modelVersionId={formattedModels[0]._versionId}
              title={"Right View"}
            />
          </div>
          <div style={{ height: '100%', width: '100%' }}          
            onClick={!this.props.enableOptimizedSelection ? this.selectEntities : undefined}>
            <IafViewerDBM 
              {...viewerProps} 
              model={formattedModels[1]}
              modelVersionId={formattedModels[1]._versionId}
              title={"Left View"}
            />
          </div>
        </CompareView>
        </div>
    );

    if (formattedModels.length == 1) return (
        <div style={{ width: '100%', height: '100%' }}>
        <CompareView mode="parallel" orientation="horizontal">
          <div style={{ height: '100%', width: '100%' }}          
            onClick={!this.props.enableOptimizedSelection ? this.selectEntities : undefined}>
            <IafViewerDBM 
              {...viewerProps} 
              title={"Right View"}
            />
          </div>
          <div style={{ height: '100%', width: '100%' }}          
            onClick={!this.props.enableOptimizedSelection ? this.selectEntities : undefined}>
            <IafViewerDBM 
              {...viewerProps} 
              title={"Left View"}
            />
          </div>

        </CompareView>
        </div>
    );

    {/* <img
      src="https://picsum.photos/800/400?random=1"
      backgroundColor="#000"
      alt="Nature did not load"
      style={{ width: '100%'
              , height: '100%'
              , objectFit: 'cover'
            }}
    />
    <img
      src="https://picsum.photos/800/400?random=2"
      backgroundColor="#000"
      alt="Tech did not load"
      style={{ width: '100%'
              , height: '100%'
              , objectFit: 'cover
            }}
    /> */}
  } else {
    return (
      <div style={{ height: '100%', width: '100%' }}
        onClick={!this.props.enableOptimizedSelection ? this.selectEntities : undefined}>
        <IafViewerDBM {...viewerProps} />
      </div>
    );
  }
}

}

export const EnhancedIafViewer = ({
  model,
  compareModel,
  isolatedEntities,
  selectedEntities,
  allEntities,
  selectedElementIds,
  enableOptimizedSelection,
  viewerResizeCanvas,
  onSelect,
  hiddenElementIds,
  workflow,
  enableGisDrawer,
  mapboxToken,
  themeColor,
  coloredElementIds,
  onHoverIconColor,
  onActiveIconColor
}) => {
  const saveSettingsCallback = useCallback((settings) => {
    localStorage.iafviewer_settings = JSON.stringify(settings);
  }, []);
  const emptyArray = useMemo(() => [], []);
  const viewerSettings = useMemo(
    () =>
      localStorage.iafviewer_settings
        ? JSON.parse(localStorage.iafviewer_settings)
        : undefined,
    [localStorage.iafviewer_settings]
  );

  const modelIdsList = () => {
    if (isolatedEntities.length > 0) {
      isolatedEntities.forEach((el) => {
        modelViewerIdsList.push(el.modelViewerIds[0]);
      });

    }
  }
  const colorGroups = useMemo(() => [
    {
      groupName: "SystemColor",
      colors: [
        {
          color: EntitySelectionPanel.ThemeColor.color, //themeColor,
          opacity: 0.9,
          elementIds: modelViewerIdsList,
        },
      ],
    },

  ]);

  function handleOnIsolateElementChangeCallback(element) {
    console.log("IafViewerDbmCallbacks.Isolate Element Change:", element);
  }

  function handleOnSelectedElementChangeCallback(element) {
    console.log("IafViewerDbmCallbacks.Selected Element Change:", element);
  }

  function handleOnHiddenElementChangeCallback(element) {
    console.log("IafViewerDbmCallbacks.Hidden Element Change:", element);
  }

  function handleOnResetCallback() {
    console.log("IafViewerDbmCallbacks.Reset Callback");
  }

  function handleOnSnackbarOpenCallback(message) {
    console.log("IafViewerDbmCallbacks.Snackbar Open Callback:", message);
  }

  // Callback function triggered when configuring the 2D toolbar.
  const handle2dToolbarConfigCallback = (toolbarList, measurementList, viewList, navigationList, shadingList, manipulateList) => {
    console.log("IafViewerDbmCallbacks.2D Toolbar Configuration:", toolbarList, measurementList, viewList, navigationList, shadingList, manipulateList);
  };

  // Callback function triggered when configuring the 3D toolbar.
  const handle3dToolbarConfigCallback = (toolbarList, measurementList, viewList, navigationList, shadingList, manipulateList) => {
    console.log("IafViewerDbmCallbacks.3D Toolbar Configuration:", toolbarList, measurementList, viewList, navigationList, shadingList, manipulateList);
  };

  // Callback function triggered when configuring the default toolbar.
  const handleDefaultToolbarConfigCallback = (toolbarList, measurementList, viewList, navigationList, shadingList, manipulateList) => {
    console.log("IafViewerDbmCallbacks.Default Toolbar Configuration:", toolbarList, measurementList, viewList, navigationList, shadingList, manipulateList);
  };

  const extra = {
    model: model,
    compareModel: compareModel,
    serverUri: endPointConfig.graphicsServiceOrigin,
    hiddenElementIds: hiddenElementIds,
    viewerResizeCanvas: viewerResizeCanvas,
    settings: viewerSettings,
    saveSettings: saveSettingsCallback,
    colorGroups: colorGroups,
    modelIdsList: modelIdsList(),
    onSelect: onSelect,
    isolatedEntities,
    selectedEntities: _.isEmpty(selectedEntities)
      ? emptyArray
      : selectedEntities, //trying to avoid re-render here
    selectedElementIds,
    enableOptimizedSelection,
    btns: {
      Orthographic: { display: true },
      Analytics: { display: true },
      viewer2D: { display: true },
      Reset: { display: true },
      Projection: { display: true },
      View: { display: true },
      Shading: { display: true },
      Navigation: { display: true },
      Measurement: { display: true },
      Utilities: { display: true },
      Settings: { display: true },
    },
    units: "inch", // mm, foot, ft, inch, in, meter,m
    // cuttingPlaneValues: {
    //   min: {x: 10, y: 10, z: 10}
    // },
    isIpaDev: false,
    showToolTip: true,
    sidePanelColor: "#1D1D1D",
    toolbarColor: "#333333",
    isShowNavCube: true,
    enableFocusMode: true,
    onHoverIconColor: "var(--filter)",
    onActiveIconColor: "var(--filter)",
    toolbarSize:  "medium",
    OnIsolateElementChangeCallback:   handleOnIsolateElementChangeCallback,
    OnSelectedElementChangeCallback:  handleOnSelectedElementChangeCallback,
    OnHiddenElementChangeCallback:  handleOnHiddenElementChangeCallback,
    OnResetCallback:  handleOnResetCallback,
    OnSnackbarOpenCallback:  handleOnSnackbarOpenCallback,
    On2dToolbarConfigCallback: handle2dToolbarConfigCallback,
    On3dToolbarConfigCallback: handle3dToolbarConfigCallback,
    OnDefaultToolbarConfigCallback: handleDefaultToolbarConfigCallback,
    showNotification: true,
    allEntities,
    pdf2DUrl: "/reference/pdf/line.pdf",
    workflow,
    enableGisDrawer,
    mapboxToken: mapboxToken
  }
  return (
    <ViewerWrapper {...extra} />
  );
};