import {v4 as uuid} from "uuid";
import {
    BUILDING_OF_INTEREST, DISTRICT_COLORS,
    DISTRICT_OVERHEAD_CAMERAS,
    GIS_LAYER_NAMES,
    GIS_LAYERS,
    INITIAL_POSITION_CAMERA,
    SYMBOLOGY
} from "./config";
import { IafEvmUtils } from "@dtplatform/iaf-viewer";
// import { UE_TELEPORT_PADS } from "../ue/UEConfig"
// import {MMV_COMMANDS} from "@invicara/iaf-mmv";

export const clearLayerFilterAction = (layerName) => {
    const command = IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.CUSTOM,
        commandRef: uuid(),
        params: {
            // commandName: 'filtermodel',
            commandName: IafEvmUtils.MMV_COMMANDS.SLICE_ELEMENTS,
            params: {
                clear: true,
                extra: {
                    layerNames : [layerName]
                }
            }
        }
    });
    return [command];
}

export const clearFilterDistrictAction = () => {
    return [...clearLayerFilterAction(GIS_LAYER_NAMES.DISTRICTS),...clearLayerFilterAction(GIS_LAYER_NAMES.BUILDINGS)];
}

export const filterDistrictAction = (districtName) => {
    console.log ('PlatformReferenceApp.Tourism.Commands.filterDistrictAction', districtName);
    //zoom to its geometry and setup feature filters to mask out other features
    const command = IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.CUSTOM,
        commandRef: uuid(),
        params: {
            // commandName: 'filtermodel',
            commandName: IafEvmUtils.MMV_COMMANDS.SLICE_ELEMENTS,
            params: {
                clear: false,
                ids: [districtName],
                extra: {
                    layerNames : [GIS_LAYER_NAMES.DISTRICTS],
                    field: GIS_LAYERS[GIS_LAYER_NAMES.DISTRICTS].idField,
                }
            }
        }
    });
    const command2 = IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.CUSTOM,
        commandRef: uuid(),
        params: {
            // commandName: 'filtermodel',
            commandName: IafEvmUtils.MMV_COMMANDS.SLICE_ELEMENTS,
            params: {
                clear: false,
                ids: [districtName],
                extra: {
                    layerNames : [GIS_LAYER_NAMES.BUILDINGS],
                    field: GIS_LAYERS[GIS_LAYER_NAMES.BUILDINGS].relatedLayersField[GIS_LAYER_NAMES.DISTRICTS],
                }
            }
        }
    });
    return [command,command2];
}

export const zoomToOnly = (districtName) => {
    return IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.ZOOM_TO,
        commandRef: uuid(),
        params: {
            elementId: districtName,
            extra: {
                field: GIS_LAYERS[GIS_LAYER_NAMES.DISTRICTS].idField,
                layerNames: [GIS_LAYER_NAMES.DISTRICTS],
            }
        }
    });
}

export const setCameraAction = (params) => {
    const commands = [IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
            commandName: IafEvmUtils.MMV_COMMANDS.SET_CAMERA,
            commandRef: uuid(),
            params: params
        })
    ];
    //zoom to its geometry and setup feature filters to mask out other features
    return commands;
}

export const zoomToInitAction = () => {
    const commands = setCameraAction(INITIAL_POSITION_CAMERA);
    //zoom to its geometry and setup feature filters to mask out other features
    return commands;
}

export const zoomToDistrictAction = (districtName) => {
    if(!districtName){
        return null;
    }
    const commands = [
        ...filterDistrictAction(districtName),
        IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
            commandName: IafEvmUtils.MMV_COMMANDS.ZOOM_TO,
            commandRef: uuid(),
            params: {
                elementId: districtName,
                extra: {
                    field: GIS_LAYERS[GIS_LAYER_NAMES.DISTRICTS].idField,
                    layerNames: [GIS_LAYER_NAMES.DISTRICTS],
                }
            }
        }),
        ...sketchBuildingsAction(districtName)
    ];
    //zoom to its geometry and setup feature filters to mask out other features
    return commands;
}

export const zoomToDistrictOverheadAction = (districtName) => {
    if(!districtName){
        return null;
    }
    const commands = [
        ...filterDistrictAction(districtName),
        ...setCameraAction(DISTRICT_OVERHEAD_CAMERAS[districtName]),
        ...sketchBuildingsAction(districtName, true)
    ];
    //zoom to its geometry and setup feature filters to mask out other features
    return commands;
}

export const applyDistrictSymbologyAction = (districtName) => {
    const symbol = SYMBOLOGY.DISTRICTS[districtName].symbolSource;
    const commands = IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.ADD_GRAPHICS,
        commandRef: uuid(),
        params: {
            graphics: [{
                type: "symbol",
                id: `label_point-${districtName}`,
                symbolType: 'custom',
                symbolSource: JSON.parse(JSON.stringify(symbol)), //i.e. MusicCIM
                text: districtName,
                point: SYMBOLOGY.DISTRICTS[districtName].point,
                //symbolColor: [230.0/255.0,196.0/255.0,181.0/255.0,1]//[0.9,0,0,1]
            },
            ]
        }
    });
    return commands;
}

export const applyDistrictsSymbologyAction = () => {
    const symbology = Object.entries(SYMBOLOGY.DISTRICTS).map(([k,v])=>({
        type: "symbol",
        id: `label_point-${k}`,
        symbolType: 'custom',
        symbolSource: JSON.parse(JSON.stringify(v.symbolSource)), //i.e. MusicCIM
        text: k,
        point: v.point,
        //symbolColor: [230.0/255.0,196.0/255.0,181.0/255.0,1]//[0.9,0,0,1]}
    }));
    if(!symbology || symbology.length==0){
        return [];
    }
    const commands = IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.ADD_GRAPHICS,
        commandRef: uuid(),
        params: {
            graphics: symbology

    }});
    return commands;
}

export const getDistrictGraphics = () => {
    const commands = applyDistrictsSymbologyAction();
    return commands;
    // return commands[0].params.graphics;    
    // const graphicsCommands = [];
    // const commandName = commands[0].commandName;
    // commands[0].params.graphics.forEach(cmd=>graphicsCommands.push({commandName, params: { graphics: [cmd]}}));
    // return graphicsCommands;
}

export const clearSymbologyAction = () => {
    /*const commands = [{
        commandName: IafEvmUtils.MMV_COMMANDS.REMOVE_GRAPHICS,
        commandRef: uuid(),
        params: {
            ids: Object.keys(SYMBOLOGY.DISTRICTS).map(districtName=>`label_point-${districtName}`)
        }
    }];*/
    const commands = IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.ADD_GRAPHICS,
        commandRef: uuid(),
        params: {
            removeAll: true,
            graphics: []
        }
    });
    return commands;
}

export const applyBuildingSymbologyAction = (districtName) => {
    const buildings = Object.entries(SYMBOLOGY.BUILDINGS).filter(([k,b])=>b.district===districtName);
    if(!buildings || buildings.length==0){
        return [];
    }
    const commands = IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.ADD_GRAPHICS,
        commandRef: uuid(),
        params: {
            graphics: buildings
                .map(([bn,v])=> ({
                    type: "symbol",
                    id: `label_point-building-${bn}`,
                    symbolType: 'custom',
                    symbolSource: JSON.parse(JSON.stringify(SYMBOLOGY.BUILDINGS[bn].symbolSource)), //i.e. MusicCIM
                    text: bn,
                    point: v.point,
                    //symbolColor: [230.0/255.0,196.0/255.0,181.0/255.0,1]//[0.9,0,0,1]}
            }))

        }});
    return commands;
}

export const clearBuildingSymbologyAction = (districtName) => {
    console.log("clearBuildingSymbologyAction: "+districtName);
    const buildings = Object.entries(SYMBOLOGY.BUILDINGS).filter(([k,b])=>b.district===districtName);
    if(!buildings || buildings.length==0){
        return [];
    }
    const commands = [{
        commandName: IafEvmUtils.MMV_COMMANDS.REMOVE_GRAPHICS,
        commandRef: uuid(),
        params: {
            ids: buildings.map(([bn,v])=> `label_point-building-${bn}`)
        }
    }];
    return commands;
}

export const sketchBuildingsAction = (districtName, themePhotoTours = false) => {

    // WHAT:
    // When we've zoomed into a district we want to color any "buildings of interest" in the districts main color.
    // All the other buildings will be mostly transparent and use a "sketch" renderer for the lines
    // For the inset map overhead view we also want to theme teleport pads with a blue sketchline and photo tours
    // with a green sketch line.

    // HOW:
    // 1. create a group of buildings of interest and set their color to main color
    // 2. add the 'other' object with the renderer for the "uninteresting buildings"
    // 3. rather than using the GIS Config to figure out what layers to look in we provide the "extra" object
    //    which tells the GIS system what layer and field name to use.  We could leave "extra" off and the 
    //    GIS system would then use the info in the config file to figure it out.

    let boiIds = BUILDING_OF_INTEREST.filter(b=>b.district===districtName).map(b=>b.osm_id);
    if(boiIds.length===0){
        boiIds = ['']; //has to have value, otherwise expression breaks and will clear themeing
    }

    let params = {
        // 1. one group of buildings of interest that will be in the main color
        groups: [{ color: DISTRICT_COLORS[districtName].mainColor, ids: boiIds }],
        // 2. all others will be transparent with an edge sketch renderer
        other: { 
            color: [235, 235, 235],
            opacity: 0.1,
            edges: {
                width: 1,
                color: '#cccccc',
                opacity: 0.1,
                extra: {
                    extensionLength: 2,
                    type: 'sketch'
                }
            }
        },
        // 3. tell the GIS system which layer and which field to look at
        extra: {
            field: GIS_LAYERS[GIS_LAYER_NAMES.BUILDINGS].idField,
            fieldType: GIS_LAYERS[GIS_LAYER_NAMES.BUILDINGS].idFieldType || "String",
            layerNames: [GIS_LAYER_NAMES.BUILDINGS]
        }
    }

    if (themePhotoTours) {
        let tourIds = BUILDING_OF_INTEREST.filter(b=>b.hasPhotoTour===true).map(b=>b.osm_id);
        if (tourIds.length==0) tourIds = ['']
        else {
            let s = new Set(tourIds)
            params.groups[0].ids = boiIds.filter(id => !s.has(id))
        }
        params.groups.push({ 
            ids: tourIds,
            color: DISTRICT_COLORS[districtName].mainColor,
            edges: { width: 1, color: '#ffff00', opacity: 0.5 }
        })



        // 
        // NO LONGER THEMING TELEPORT PADS
        //
        // let padIds = UE_TELEPORT_PADS.map(id => id.split("_")[1])
        // if (padIds.length==0) padIds = ['']
        // else {
        //     let s = new Set(tourIds.concat(boiIds))
        //     padIds = padIds.filter(id => !s.has(id))
        // }
        // let color = DISTRICT_COLORS[districtName].mainColor.slice(0)
        // color[3] = 25
        // params.groups.push({ 
        //     ids: padIds,
        //     color: [235, 235, 235],
        //     opacity: 0.1,
        //     edges: {
        //         width: 1,
        //         color: '#ffffff',
        //         opacity: 0.3
        //     }
        // })
    }



    const commands = [IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.THEME_ELEMENTS,
        commandRef: uuid(),
        params
    })];
    return commands;
}

export const sketchBuildingsClearAction = () => {
    const commands = [IafEvmUtils.apiUnwrapMmvCommadToEvmProperty({
        commandName: IafEvmUtils.MMV_COMMANDS.THEME_ELEMENTS,
        commandRef: uuid(),
        params: {
            clear: true,
            extra: {
                layerNames: [GIS_LAYER_NAMES.BUILDINGS]
            }
        }
    })];
    return commands;
}

