import BuildingCimSymbol from './gis_symbols/BuildingCimSymbol'
import HotelCimSymbol from './gis_symbols/HotelCimSymbol'
import BarCimSymbol from './gis_symbols/BarCimSymbol'
import TheaterCimSymbol from './gis_symbols/TheaterCimSymbol'
import BasicDistrictSymbol from './gis_symbols/BasicDistrictSymbol'
import ForestCimSymbol from './gis_symbols/ForestCimSymbol'
import VesselCimSymbol from './gis_symbols/VesselCimSymbol'
import ShoppingCimSymbol from './gis_symbols/ShoppingCimSymbol'
import FoodCimSymbol from './gis_symbols/FoodCimSymbol'
import BusinessCimSymbol from './gis_symbols/BusinessCimSymbol'
import MusicCimSymbol from './gis_symbols/MusicCimSymbol'
import {generateColorCIMSymbolCopy} from "./gis_symbols/helpers";

/************************************************************************************************
 * @buildingOfInterest Building Of Interest
 options.buildingOfInterest = [
 // first five are for the tourism example. Not setting numberOfTickets causes a minor change in the symbology in the map
 // id = Building Feature OID from data
 // name = Building feature name in data
 // district = District name in data (building data)
 // symbology = CimSymbol, has some specific rules for use - rules are in the GISViewer code
 // {id: 104296, name: 'Chicago Title & Trust Building', district:'Theatre District', symbology: HotelCimSymbol},
 // {id: 104349, name: 'Randolph Tower', district:'Theatre District', symbology: BarCimSymbol},
 // {id: 105192, name: 'Dublin Building', district:'Theatre District', symbology: HotelCimSymbol},
 // {id: 104811, name: 'Cadillac Palace Theatre', district:'Theatre District', symbology: TheaterCimSymbol},
 // {id: 104126, name: 'Kemper Building', district:'Theatre District', symbology: HotelCimSymbol}

 // These are the operations (ticket) example. Setting  numberOfTickets causes a minor change in the symbology in the map
 // id = Building Feature OID from data
 // name = Building feature name in data
 // district = District name in data (building data)
 // symbology = CimSymbol, has some specific rules for use - rules are in the GISViewer code
 // refinedBuildingLayer: Features to display instead of the feature listed in this object
 {id: 104296, name: 'Chicago Title & Trust Building', district:'Theatre District', numberOfTickets: 1, symbology: BuildingCimSymbol},
 {id: 104349, name: 'Randolph Tower', district:'Theatre District', numberOfTickets: 2, symbology: BuildingCimSymbol},
 {id: 105192, name: 'Dublin Building', district:'Theatre District', numberOfTickets: 3, symbology: BuildingCimSymbol, refinedBuildingLayer: 'GeoBIM Franklin Building - GeoBIM Franklin Building'},
 {id: 104811, name: 'Cadillac Palace Theatre', district:'Theatre District', numberOfTickets: 4, symbology: BuildingCimSymbol},
 {id: 104126, name: 'Kemper Building', district:'Theatre District', numberOfTickets:5, symbology: BuildingCimSymbol}
 ]

 * @districtsLayer District Layer
 // Required for both tourism and operations example
 // title = SceneLayer or FeatureLayer title property from ArcGIS service
 // districtField = district field in data, in FeatureLayer from ArcGIS service
 options.districtsLayer = {title:'Chicago POC Districts', districtField: 'HL_DistrictName'} //, labelfield: "name"}

 * @districtLabelPnts District Label Points
 // districtLabelPnts are the label and symbols above the districts.
 // Use this format for Operations mode
 // title = SceneLayer title property from ArcGIS service
 // symbology = CimSymbol used for all districts in the SceneLayer from ArcGIS Service
 // districtField = district field in data, in SceneLayer's FeatureLayer from ArcGIS service
 options.districtLabelPnts = {title: 'Chicago Districts Label Points - Chicago Districts Centers 3d label height', symbology: BasicDistrictSymbol, districtField: 'HL_DistrictName'}

 // Use this format for Tourism mode
 // title = SceneLayer title property from ArcGIS service
 // symbology = Assigns different symbols (CimSymbols) to districts in SceneLayer from ArcGIS service
 // districtField = district field in data, in SceneLayer's FeatureLayer from ArcGIS service
 // options.districtLabelPnts = {title: 'Chicago Districts Label Points - Chicago Districts Centers 3d label height',
    //     symbology: {
    //         symbols: [
    //             {name: "Downtown Business District", symbology: BusinessCimSymbol},
    //             {name: "Magnificient Mile", symbology: ShoppingCimSymbol},
    //             {name: "Millenium Park", symbology: ForestCimSymbol},
    //             {name: "South Loop", symbology: MusicCimSymbol},
    //             {name: "Streeterville", symbology: VesselCimSymbol},
    //             {name: "Theatre District", symbology: TheaterCimSymbol},
    //             {name: "West Loop", symbology: FoodCimSymbol}
    //         ]
    //     }, districtField: 'HL_DistrictName'}

 * @buildings3Dlyr Building 3D Layer
 // Required for both tourism and operations modes
 // title = SceneLayer title property from ArcGIS service
 // districtField = district field in data, in SceneLayer's FeatureLayer from ArcGIS service
 // heightField = field with elevation info in SceneLayer's FeatureLayer from ArcGIS service. Used only in popup
 // buildingNameField = field with building name in SceneLayer's FeatureLayer from ArcGIS service. Used only in popup
 // identifyOnHover = when true, show popup during onHover event for buildings
 options.buildings3Dlyr = {title:'Chicago 3D Buildings - City of Chicago 3D', districtField: 'hl_district', heightField: 'hl_bldght', buildingNameField: 'name', identifyOnHover: false}

 * @buildingsLabelPnts Building Layer Points
 // Required for both tourism and operations modes
 // title = title property from ArcGIS service
 // name = field with building name in FeatureLayer from ArcGIS service
 options.buildingsLabelPnts = {title: 'ChicagoBuildingPoints_Districts', buildingNameField: 'name'}

 *
 */

export const GIS_LAYER_NAMES = {

    BUILDINGS : "District Buildings - ChicagoBuildingsInDistricts",
    DISTRICTS: "Chicago POC Districts",
    BUILDINGS_LABEL_POINTS: "Chicago Building Label Points with Districts - Chicago Building Label Points with Districts",
    DISTRICTS_LABEL_POINTS: "Chicago Districts Label Points - Chicago Districts Centers 3d label height",
    EXCHANGE_BUILDING: "Exchange Building - Exchange Building",
    BULDINGS_POINTS: "ChicagoBuildingPoints_Districts",
}

export const GIS_DISTRICT_NAMES = {
    DOWNTOWN: "Downtown Business District",
    MILE: "Magnificient Mile",
    PARK: "Millenium Park",
    SLOOP: "South Loop",
    STREETV: "Streeterville",
    THEATRE: "Theatre District",
    WLOOP: "West Loop"
}

export const GIS_BUILDING_NAMES = {
    TT: 'Chicago Title & Trust Building',
    Randolph:'Randolph Tower',
    Dublin: 'Dublin Building',
    Cadillac: 'Cadillac Palace Theatre',
    Kemper: 'Kemper Building'
}

const DISTRICT_CAMERAS = {
    [GIS_DISTRICT_NAMES.THEATRE]:
        { position: { x: -87.645, y: 41.873, z: 1350.360 }, rotation: { yaw: 41.182, pitch: 54.541 } },
    [GIS_DISTRICT_NAMES.WLOOP]:
        { position: { x: -87.680, y: 41.849, z: 2808.446 }, rotation: { pitch: 54.600, yaw: 41.218 } },
    [GIS_DISTRICT_NAMES.SLOOP]:
        { position: { x: -87.644, y: 41.859, z: 1461.495 }, rotation: { pitch: 54.611, yaw: 41.227 } },
    [GIS_DISTRICT_NAMES.DOWNTOWN]:
        { position: { x: -87.643, y: 41.870, z: 1121.975 }, rotation: { pitch: 54.628, yaw: 41.235 } },
    [GIS_DISTRICT_NAMES.MILE]:
        { position: { x: -87.634, y: 41.880, z: 955.145 },  rotation: { pitch: 54.639, yaw: 41.245 } },
    [GIS_DISTRICT_NAMES.STREETV]:
        { position: { x: -87.631, y: 41.878, z: 1221.017 }, rotation: { pitch: 54.651, yaw: 41.256 } },
    [GIS_DISTRICT_NAMES.PARK]:
        { position: { x: -87.647, y: 41.868, z: 1830.677 }, rotation: { pitch: 50.690, yaw: 69.502 } }
}

export const INITIAL_POSITION_CAMERA = {
    position: { x: -87.667, y: 41.863, z: 3262.319},
    rotation: { pitch: 50.063, yaw: 61.458}
}

// "osm_id" is the id used by the UE implementation of the "teleport pads" (a teleport pad is where the user
// lands after zooming to a building))
export const BUILDING_OF_INTEREST = [
    {id: 104296, osm_id: 144846547, name: GIS_BUILDING_NAMES.TT, district:GIS_DISTRICT_NAMES.THEATRE},
    {id: 104349, osm_id: 147013786, name: GIS_BUILDING_NAMES.Randolph, district:GIS_DISTRICT_NAMES.THEATRE, hasPhotoTour: true},
    {id: 105192, osm_id: 99999, name: GIS_BUILDING_NAMES.Dublin, district:GIS_DISTRICT_NAMES.THEATRE},
    {id: 104811, osm_id: 210671540, name: GIS_BUILDING_NAMES.Cadillac, district:GIS_DISTRICT_NAMES.THEATRE, hasPhotoTour: true},
    {id: 104126, osm_id: 64389514, name: GIS_BUILDING_NAMES.Kemper, district:GIS_DISTRICT_NAMES.THEATRE}
]

export const DISTRICT_OVERHEAD_CAMERAS = {
    [GIS_DISTRICT_NAMES.THEATRE]:
        {position: { x: -87.632, y: 41.885, z: 2105.556 }, rotation: { pitch: 0, yaw: 0 } },
}

const DEFAULT_HIGHLIGHT = {
    onEvent: 'pointer-move', //or 'selection_update'
    ids: undefined
}

export const DISTRICT_COLORS = {
    [GIS_DISTRICT_NAMES.DOWNTOWN]: {
        mainColor: [230, 196, 181, 255]
    },
    [GIS_DISTRICT_NAMES.MILE]: {
        mainColor: [177, 216, 212, 255]
    },
    [GIS_DISTRICT_NAMES.PARK]: {
        mainColor:[230, 230, 0, 255]
    },
    [GIS_DISTRICT_NAMES.SLOOP]: {
        mainColor: [128, 177, 211, 255]
    },
    [GIS_DISTRICT_NAMES.STREETV]: {
        mainColor: [253, 180, 98, 255]
    },
    [GIS_DISTRICT_NAMES.THEATRE]: {
        mainColor: [166, 125, 154, 255]
    },
    [GIS_DISTRICT_NAMES.WLOOP]: {
        mainColor: [214, 214, 214, 255]
    }
}

export const SYMBOLOGY = {
    DISTRICTS : {
        [GIS_DISTRICT_NAMES.DOWNTOWN]: {
            symbolSource: generateColorCIMSymbolCopy(BusinessCimSymbol, DISTRICT_COLORS[GIS_DISTRICT_NAMES.DOWNTOWN].mainColor),
            point: {x: -9755028.0996, y: 5142849.480899997, z: 375}
        },
        [GIS_DISTRICT_NAMES.MILE]: {
            symbolSource: generateColorCIMSymbolCopy(ShoppingCimSymbol, DISTRICT_COLORS[GIS_DISTRICT_NAMES.MILE].mainColor),
            point: {x: -9754329.3879, y: 5144322.482299998, z : 350}
        },
        [GIS_DISTRICT_NAMES.PARK]: {
            symbolSource: generateColorCIMSymbolCopy(ForestCimSymbol, DISTRICT_COLORS[GIS_DISTRICT_NAMES.PARK].mainColor),
            point: {x: -9753895.5236, y: 5142468.902400002, z: 225}
        },
        [GIS_DISTRICT_NAMES.SLOOP]: {
            symbolSource: generateColorCIMSymbolCopy(MusicCimSymbol,DISTRICT_COLORS[GIS_DISTRICT_NAMES.SLOOP].mainColor),
            point: {x:-9754882.0659, y : 5141763.063100003, z:225}
        },
        [GIS_DISTRICT_NAMES.STREETV]: {
            symbolSource: generateColorCIMSymbolCopy(VesselCimSymbol, DISTRICT_COLORS[GIS_DISTRICT_NAMES.STREETV].mainColor),
            point: {x:-9753635.288, y : 5144416.958999999, z:225}
        },
        [GIS_DISTRICT_NAMES.THEATRE]: {
            symbolSource: generateColorCIMSymbolCopy(TheaterCimSymbol,DISTRICT_COLORS[GIS_DISTRICT_NAMES.THEATRE].mainColor),
            point: {x: -9755129.178, y: 5143693.300399996, z: 330}
        },
        [GIS_DISTRICT_NAMES.WLOOP]: {
            symbolSource: generateColorCIMSymbolCopy(FoodCimSymbol,DISTRICT_COLORS[GIS_DISTRICT_NAMES.WLOOP].mainColor),
            point: {x:-9756321.1532, y: 5143829.9965, z: 300}
        }
    },
    BUILDINGS : {
        [GIS_BUILDING_NAMES.TT]: {
            district: GIS_DISTRICT_NAMES.THEATRE,
            symbolSource: generateColorCIMSymbolCopy(HotelCimSymbol,DISTRICT_COLORS[GIS_DISTRICT_NAMES.THEATRE].mainColor),
            point: { x : -9754978.2208, y : 5143749.817500003, z : 376.89659999999276 }

        },
        [GIS_BUILDING_NAMES.Randolph]: {
            district: GIS_DISTRICT_NAMES.THEATRE,
            symbolSource: generateColorCIMSymbolCopy(BarCimSymbol,DISTRICT_COLORS[GIS_DISTRICT_NAMES.THEATRE].mainColor),
            point: { x : -9755332.4485, y : 5143741.270800002, z : 269.7347000000009}
        },
        [GIS_BUILDING_NAMES.Cadillac]: {
            district: GIS_DISTRICT_NAMES.THEATRE,
            symbolSource: generateColorCIMSymbolCopy(TheaterCimSymbol,DISTRICT_COLORS[GIS_DISTRICT_NAMES.THEATRE].mainColor),
            point: {x : -9755265.5984, y : 5143634.017200001, z : 259.69220000000496}
        },
        [GIS_BUILDING_NAMES.Kemper]: {
            district: GIS_DISTRICT_NAMES.THEATRE,
            symbolSource: generateColorCIMSymbolCopy(HotelCimSymbol,DISTRICT_COLORS[GIS_DISTRICT_NAMES.THEATRE].mainColor),
            point: {x : -9754639.0759, y : 5143992.197999999, z : 343.6391000000003}
        }
    }
}

const SPECIFICITIES = {
    [GIS_LAYER_NAMES.DISTRICTS]: 100,
    [GIS_LAYER_NAMES.BUILDINGS]: 200,
}


const isADistrictId = (id) => {
    return GIS_DISTRICT_NAMES[id]
};

const isABuildingId = (id) => {
    return GIS_BUILDING_NAMES[id]
};

export const GIS_LAYERS = {
    [GIS_LAYER_NAMES.DISTRICTS]: {
        idField: "hl_districtname",
        isIdValidForLayer: isADistrictId,
        elementType: "district",
        highlight: {
            triggerEvent: 'pointer-move', //or 'selection_update'
            highlightableIds: undefined //undefined == all ids
        },
        specificity: SPECIFICITIES[GIS_LAYER_NAMES.DISTRICTS]
    },
    [GIS_LAYER_NAMES.BUILDINGS]: {
        idField: "osm_id",
        idFieldType: "Integer",
        // extraAttributes: ["osm_id"],
        isIdValidForLayer: isABuildingId,
        elementType: "building",
        highlight: {
            triggerEvent: 'pointer-move', //or 'selection_update'
            highlightableIds: undefined //undefined == all ids
        },
        specificity: SPECIFICITIES[GIS_LAYER_NAMES.BUILDINGS],
        relatedLayersField : {
            [GIS_LAYER_NAMES.DISTRICTS]:'hl_district'
        },
        //TODO: could packaged iaf-mmv viewer look at those and prepare layer popupTemplate?
        outFields: ["*"],
        popupTemplate: {
            title: `{name}`,
            content: [{
                type: "fields",
                fieldInfos: [{
                    fieldName: 'hl_district',
                    label: "District"
                }, {
                    fieldName: 'hl_bldght',
                    label: "Building Height",
                    format: {
                        places: 2,
                        digitSeparator: true
                    }
                }]
            }]
        }
        //TODO: end
    }
};

export const GIS_CONFIG = {
    // required...
    portalUrl: "https://gis.hlplanning.com/arcgis", // gis server
    apiKey:    `AAPK6c81d14b0cea426f81fa5b6abb9aa0dfGZFEkccrQXsiqRmnLr1Lpk6Khj1H4cNisCMVfDH_x313UuFSGDinndbc3hr7fqVu`, // api key for above server
    model:     "6745f5a0f1df4812a26925c19fdab6f8", // web scene id
    sceneWKID: 102100,
    // optional...
    shadows: false,
    groundOpacity: 1,
    popUpEnabled: false,           // use GIS built in popups?
    highlightOnClick: false,       // use GIS built in highlighting of elements?
    highlightTheme: {
        color: [255, 255, 0, 1],
        haloColor: "white",
        haloOpacity: 0.9,
        fillOpacity: 0.2,
        shadowColor: "black",
        shadowOpacity: 0.5
    },
    // Camera position coordinates (and coordinates sent to the MMV command) are dependent on the model's coord system
    // It is up to the controlling app (this app) to convert between model coordinate systems
    initialCameraPosition: INITIAL_POSITION_CAMERA,
    hiddenLayers: [
        GIS_LAYER_NAMES.BUILDINGS_LABEL_POINTS,
        GIS_LAYER_NAMES.DISTRICTS_LABEL_POINTS
    ],
    layerInfo: GIS_LAYERS
}
const baseConfig = GIS_CONFIG;

const cityDistrictConfig = {
    ...baseConfig,
    specificityMode: SPECIFICITIES[GIS_LAYER_NAMES.DISTRICTS],
    components: ["zoom", "compass"],
    showBasemapGallery: true
}

const insetMapConfig = {
    ...baseConfig,
    specificityMode: SPECIFICITIES[GIS_LAYER_NAMES.DISTRICTS],
    showBasemapGallery: false
}

export function getConfig({
                              configName,
                              searchParamPage = null,
                              searchParamDistrict = null,
                              blockedInputEvents = null
                          }) {
    switch(configName.toLowerCase()) {
            case 'citydistrict':
                return {
                    ...cityDistrictConfig,
                    specificityMode: searchParamPage === "district" ? SPECIFICITIES[GIS_LAYER_NAMES.BUILDINGS] : SPECIFICITIES[GIS_LAYER_NAMES.DISTRICTS],
                    initialCameraPosition: searchParamPage === "district" ? DISTRICT_CAMERAS[searchParamDistrict] : INITIAL_POSITION_CAMERA,
                    blockedInputEvents
                }
            case 'insetmapcitydistrict':
                return {
                    ...insetMapConfig,
                    specificityMode: SPECIFICITIES[GIS_LAYER_NAMES.DISTRICTS],
                    initialCameraPosition: INITIAL_POSITION_CAMERA,
                    blockedInputEvents
                }
            case 'insetmapstreetinside':
                return {
                    ...insetMapConfig,
                    specificityMode: SPECIFICITIES[GIS_LAYER_NAMES.BUILDINGS],
                    initialCameraPosition: DISTRICT_OVERHEAD_CAMERAS[searchParamDistrict],
                    blockedInputEvents
                }
            default:
                console.error("Unknown config name: " + configName)
        }
}