import React, { useEffect, useRef } from "react";

export const frequencyMap = {
  single: 1, // Single event can be triggered once
  low: 2,    // Low can trigger 2 times per minute
  medium: 5, // Medium can trigger 5 times per minute
  high: 10,  // High can trigger 10 times per minute
};

const RandomReminder = ({ events, onRemind }) => {
  const triggeredEventsRef = useRef(new Map());
  const lastTriggeredRef = useRef(new Map());

  // Function to trigger a random event and notify parent
  const triggerReminder = () => {
    if (events.length === 0) {
      console.warn("No events available.");
      return;
    }

    // Filter available events based on frequency and time since last triggered
    const availableEvents = events.filter(event => {
      const triggeredCount = triggeredEventsRef.current.get(event.id) || 0;
      const lastTriggered = lastTriggeredRef.current.get(event.id) || 0;
      const now = Date.now();
      const frequencyInMillis = (60 * 1000) / frequencyMap[event.frequency]; // Frequency in milliseconds

      // Determine if the event can still be triggered
      if (event.frequency === 'single') {
        return triggeredCount < frequencyMap[event.frequency]; // Single events only allow one trigger
      } else {
        return triggeredCount < frequencyMap[event.frequency] && (now - lastTriggered) >= frequencyInMillis;
      }
    });

    if (availableEvents.length === 0) {
      console.warn("No available events to trigger.");
      return;
    }

    // Pick a random event from the available events
    const randomEvent = availableEvents[Math.floor(Math.random() * availableEvents.length)];

    // Call the parent's callback function with the random event
    onRemind(randomEvent);

    // Mark the event as triggered
    triggeredEventsRef.current.set(randomEvent.id, (triggeredEventsRef.current.get(randomEvent.id) || 0) + 1);
    lastTriggeredRef.current.set(randomEvent.id, Date.now()); // Update last triggered time
  };

  // Function to setup continuous checking for available events
  const setupEventChecking = () => {
    const checkInterval = 3000; // Check every second
    setInterval(() => {
      triggerReminder();
    }, checkInterval);
  };


  // Function to reset triggered counts for low, medium, and high frequency events
  const resetTriggeredCounts = () => {
    events.forEach(event => {
      if (event.frequency !== "single") { // Reset for low, medium, high frequencies
        triggeredEventsRef.current.set(event.id, 0);
      }
    });
    console.log("Triggered counts reset for low, medium, and high frequency events.");
  };
  
  // Reset counts every minute
  setInterval(() => {
    resetTriggeredCounts();
  }, 60 * 1000); // Reset every minute
    
  useEffect(() => {
    setupEventChecking();

    // Clean up the interval when the component unmounts
    return () => clearInterval();
  }, [events]); // Depend on events and onRemind

  return null; // No UI from this component itself
};

export default RandomReminder;
