/**
 * ****************************************************************************
 *
 * INVICARA INC CONFIDENTIAL __________________
 *
 * Copyright (C) [2012] - [2023] INVICARA INC, INVICARA Pte Ltd, INVICARA INDIA
 * PVT LTD All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Invicara Inc and its suppliers, if any. The intellectual and technical
 * concepts contained herein are proprietary to Invicara Inc and its suppliers
 * and may be covered by U.S. and Foreign Patents, patents in process, and are
 * protected by trade secret or copyright law. Dissemination of this information
 * or reproduction of this material is strictly forbidden unless prior written
 * permission is obtained from Invicara Inc.
 */

import React from "react";
import TreeView from "../../helpers/TreeView";
import Select from "react-select";
import InfoIcon from "@mui/icons-material/Info";
import common from "../../helpers/common";
import "./RestConnectors.css";
import { IafDataSource } from "@dtplatform/platform-api";
import { IafScriptEngine } from "@dtplatform/iaf-script-engine";
import "@dtplatform/iaf-lib/dist/iaf-lib.css";
import { GenericMatButton } from "@dtplatform/platform-app-conflux/modules/IpaControls";

class RestConnector extends React.Component {
  constructor(props) {
    super(props);
    this.state = {
      treeData: null,
      selectedValue: null,
      options: [],
      selectedOption: "",
      selectedOptionValues: [],
      showInfo: false,
    };
    this.createRestConnectors = this.createRestConnectors.bind(this);
  }

  // Function to delete existing REST connector
  async deleteExistingRestConnectors(ctx, _namespaces) {
    const query = { _namespaces, _name: "project_rest_connector" };
    const orchestrators = await IafScriptEngine.getDatasources(query, ctx);

    // Iterate over orchestrators and delete each REST connector
    for (const orch of orchestrators) {
      if (orch._name === "project_rest_connector") {
        await IafDataSource.deleteOrchestrator(orch.id, ctx);
      }
    }
  }

  // Function to add a new REST connector
  async addRestConnector(ctx, configUrl, token) {
    // Parameters for the new REST connector
    const restConnectorParams = {
      _orchcomp: "rest_connector",
      _name: "Copy assets relationships from previous version",
      _sequenceno: 1,
      _actualparams: {
        _scriptName: "restConnectors",
        _url: `${configUrl.itemServiceOrigin}/itemsvc/api/v1/nameduseritems?nsfilter=${ctx._namespaces[0]}`,
        _auth: {
          _type: "BearerToken",
          _params: {
            _token: token,
          },
        },
        to: "json",
      },
    };

    // Adding the new REST connector
    const restConnector = await IafScriptEngine.addDatasource({
      _name: "project_rest_connector",
      _description: "Tester",
      _namespaces: [ctx._namespaces[0]],
      _userType: "PROJECT_REST_CONNECTOR",
      _instant: true,
      _params: { tasks: [restConnectorParams] },
    });

    return restConnector;
  }

  // Function to get the result from the REST connector
  async getRestConnectorResult(ctx) {
    const query = {
      _namespaces: ctx._namespaces,
      _name: "project_rest_connector",
    };
    const orchestrators = await IafScriptEngine.getDatasources(query, ctx);

    // Iterate over orchestrators to find the REST connector and get the result
    for (const orch of orchestrators) {
      if (orch._name === "project_rest_connector") {
        const req = { orchestratorId: orch.id };
        const namedUserItemList = await IafScriptEngine.runDatasource(req, ctx);
        return namedUserItemList._result.rest_conn_result._list;
      }
    }

    return []; // Return an empty array if no result found
  }

  // Main function to create REST connectors
  async createRestConnectors() {
    const { project, ctx, token, configUrl, _namespaces } =
      await common.getCurrentProjectInfo();

    // Delete existing REST connectors
    await this.deleteExistingRestConnectors(ctx, ctx._namespaces);

    // Add a new REST connector
    await this.addRestConnector(ctx, configUrl, token);

    // Get the result from the REST connector and update state
    const result = await this.getRestConnectorResult(ctx);
    this.setState({ options: result });
    this.setState({ selectedValue: result });
  }

  handleOptionChange = (event) => {
    const { options } = this.state;
    const selectedOption = event.value;
    const selectedOptionValues = options.find(
      (option) => option._name === selectedOption
    );

    this.setState({
      selectedOption,
      selectedOptionValues,
      treeData: selectedOptionValues,
    });
  };

  toggleInfo = () => {
    this.setState((prevState) => ({
      showInfo: !prevState.showInfo,
    }));
  };

  renderInfo() {
    const { showInfo } = this.state;

    return (
      showInfo && (
        <>
          <ul className="bullet-list">
            <li>
              Configure the REST connector with the appropriate URL and
              authentication.
            </li>
            <li>
              Efficiently make HTTP requests and retrieve data from external
              sources.
            </li>
            <li>
              Seamlessly pass the data to the next step in your application or
              workflow.
            </li>
          </ul>
          <ul>
            <li>
              When the "Generic REST Connectors" button is clicked, the
              application utilizes Bearer Token authentication to retrieve a
              response from the nameduseritems.
            </li>
            <li>
              To view the details of a specific nameduseritem, simply select it
              from the dropdown menu.
            </li>
          </ul>
        </>
      )
    );
  }

  renderButton() {
    return (
      <div style={{ textAlign: "center", marginTop: "21px" }}>
        <GenericMatButton
          onClick={this.createRestConnectors}
          styles={{ marginRight: "15px" }}
        >
          Generic Rest Connectors
        </GenericMatButton>
      </div>
    );
  }

  renderTree() {
    const {
      treeData,
      selectedValue,
      selectedOption,
      options,
      selectedOptionValues,
    } = this.state;

    const selectStyles = {
      width: "-webkit-fill-available",
      borderWidth: "medium",
      control: (provided) => ({
        ...provided,
        borderColor: "var(--app-accent-color)",
        boxShadow: "0px 0px 1px var(--app-accent-color) !important",
        "&:hover": {
          borderColor: "var(--app-accent-color) !important",
          boxShadow: "0px 0px 1px var(--app-accent-color) !important",
        },
      }),
      option: (provided, state) => ({
        ...provided,
        backgroundColor: state.isSelected
          ? "var(--fancytree-one-channel-color)"
          : "white",
        "&:hover": {
          backgroundColor: state.isSelected
            ? "var(--fancytree-one-channel-color)"
            : "var(--fancytree-one-channel-color)",
        },
      }),
    };

    return (
      selectedValue && (
        <div className="tree-view">
          <div>
            <Select
              value={selectedOption}
              onChange={this.handleOptionChange}
              options={options.map((option) => ({
                value: option._name,
                label: option._name,
              }))}
              placeholder={
                selectedOption
                  ? selectedOption
                  : "Select a Named User Item to view details"
              }
              styles={selectStyles}
            />
            {treeData && (
              <div style={{ marginTop: "10px" }}>
                <TreeView
                  treeData={treeData}
                  title={selectedOptionValues._name}
                />
              </div>
            )}
          </div>
        </div>
      )
    );
  }

  render() {
    return (
      <div
        className="tableContainer"
        style={{ paddingLeft: "25%", paddingRight: "25%", paddingTop: "20px" }}
      >
        <div className="restContainer" style={{ position: "inherit" }}>
          <h2>
            Generic REST Connectors
            <InfoIcon
              className="info-icon"
              onClick={this.toggleInfo}
              style={{ cursor: "pointer", paddingRight: "7px" }}
              fontSize="large"
            />
          </h2>
          {this.renderInfo()}
        </div>
        {this.renderButton()}
        {this.renderTree()}
      </div>
    );
  }
}
export default RestConnector;
