/**
 * ****************************************************************************
 *
 * INVICARA INC CONFIDENTIAL __________________
 *
 * Copyright (C) [2012] - [2023] INVICARA INC, INVICARA Pte Ltd, INVICARA INDIA
 * PVT LTD All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Invicara Inc and its suppliers, if any. The intellectual and technical
 * concepts contained herein are proprietary to Invicara Inc and its suppliers
 * and may be covered by U.S. and Foreign Patents, patents in process, and are
 * protected by trade secret or copyright law. Dissemination of this information
 * or reproduction of this material is strictly forbidden unless prior written
 * permission is obtained from Invicara Inc.
 */

import React from "react";
import _ from "lodash";

import {
  IafPassSvc,
  IafProj,
  IafUserGroup,
} from "@dtplatform/platform-api";
import { StackableDrawer } from "./IpaDialogs/StackableDrawer";
import RadioButtons from "./IpaControls/RadioButtons";
import SimpleTextThrobber from "./IpaControls/SimpleTextThrobber";

import { GroupCard } from "./GroupCard";
import { UserCard } from "./UserCard";
import { InviteCard } from "./InviteCard";
import { InviteForm } from "./InviteForm";

import { UserGroupPermissionTable } from "./UserGroupPermissionTable";

import "./UserGroupView.scss";

class UserGroupView extends React.Component {
  constructor(props) {
    super(props);

    this.state = {
      pageModes: ["UserGroups", "Users"],
      pageMode: "UserGroups",
      userGroupModes: ["Users/Invites", "Permissions"],
      userGroupMode: "Users/Invites",
      allUserGroupNamesUpper: [], //list of uppercase usergroup names for easy comparisons
      invalidUserGroups: [], //userGroups on the project the user can't interact with
      userGroups: [], //userGroups on the project the user can interact with
      selectedUserGroup: null, //the currently selected UserGroup
      editingUserGroup: false, //if the user is editng the UserGroup
      userGroupNameEdit: "", //the editable UserGroup name
      savingUserGroup: false, //if we are saving the UserGroup to the platform
      userGroupNameEditError: null, //any error with the UserGroup name
      usersInSelectedGroup: [], //user who are in the currently selected UserGroup
      loadingInvites: true, //whether we are loading invites for usergroup or user
      invitesInSelectedGroup: [], //pending invites in a group
      expiredInvitesInSelectedGroup: [], //expired invites in a group
      users: [], //list of all users in the project
      selectedUser: null, //the currently selected user
      userGroupsForSelectedUser: [], //the valid usergroups to which the selected user belongs
      invitesForSelectedUser: [], //invites for the seleted user
      expiredInvitesForSelectedUser: [], //expired invites for the selected user
    };

    this._loadAsyncData = this._loadAsyncData.bind(this);
    this.onModeChange = this.onModeChange.bind(this);
    this.onUserGroupModeChange = this.onUserGroupModeChange.bind(this);
    this.getAllUserGroups = this.getAllUserGroups.bind(this);
    this.setSelectedUserGroup = this.setSelectedUserGroup.bind(this);
    this.toggleUserGroupEditable = this.toggleUserGroupEditable.bind(this);
    this.onUserGroupNameChange = this.onUserGroupNameChange.bind(this);
    this.updateUserGroup = this.updateUserGroup.bind(this);
    this.sortGroupsBasedOnConfig = this.sortGroupsBasedOnConfig.bind(this);
    this.loadUserGroupData = this.loadUserGroupData.bind(this);
    this.getAllUsers = this.getAllUsers.bind(this);
    this.setSelectedUser = this.setSelectedUser.bind(this);
    this.loadUserData = this.loadUserData.bind(this);
    this.updateCurrentView = this.updateCurrentView.bind(this);
    this.onCancelInvite = this.onCancelInvite.bind(this);
    this.onResendInvite = this.onResendInvite.bind(this);
    this.onAcceptInvite = this.onAcceptInvite.bind(this);
    this.canRemoveUser = this.canRemoveUser.bind(this);
    this.removeUser = this.removeUser.bind(this);
  }

  async componentDidMount() {
    const { handler, onLoadComplete } = this.props;
    const { config } = handler;

    let pageMode = "UserGroups"; // Default value

    if (config.defaultView && config.defaultView.toUpperCase() === "USERS") {
      pageMode = "Users";
    }

    // Set the state and then call _loadAsyncData
    this.setState({ pageMode }, () => {
      this._loadAsyncData();
      onLoadComplete();
      console.log("props", this.props);
      console.log("state", this.state);
    });
  }

  async _loadAsyncData() {
    await this.getAllUserGroups();
    this.getAllUsers();
  }

  onModeChange(e) {
    this.setState({ pageMode: e.target.value, editingUserGroup: false });
    if (e.target.value === "UserGroups") this.loadUserGroupData();
  }

  onUserGroupModeChange(e) {
    this.setState({ userGroupMode: e.target.value }, this.loadUserGroupData);
  }

  sortGroupsBasedOnConfig(groups, configs) {
    // Determines whether a group has a config matching one in configs
    function hasMatchingConfig(group, configs) {
      return (
        group._userAttributes.userConfigs &&
        group._userAttributes.userConfigs.some((config) =>
          configs.some((c) => c._id === config._id)
        )
      );
    }

    // Sort groups alphabetically by name
    groups.sort((a, b) => a._name.localeCompare(b._name));

    // Filter groups with matching configs for the application
    const validGroups = groups.filter((group) =>
      hasMatchingConfig(group, configs)
    );

    // Filter groups without matching configs for the application
    const invalidGroups = groups.filter(
      (group) =>
        !group._userAttributes.userConfigs || !hasMatchingConfig(group, configs)
    );

    return { validGroups, invalidGroups };
  }

  async getAllUserGroups(selectedUserGroup) {
    return new Promise((resolve, reject) => [
      IafProj.getUserGroups(this.props.selectedItems.selectedProject).then(
        async (groups) => {
          //get all user configs int he project for this application (ie. _userType)
          IafProj.getUserConfigs(this.props.selectedItems.selectedProject, {
            query: {
              _userType: this.props.selectedItems.ipaConfig.configUserType,
            },
          }).then((userConfigs) => {
            //get valid and invalid groups
            let sortedUserGroups = this.sortGroupsBasedOnConfig(
              groups,
              userConfigs
            );

            let allUserGroupNamesUpper = groups.map((g) =>
              g._name.toUpperCase()
            );

            let applySelectedUserGroup = selectedUserGroup
              ? selectedUserGroup
              : sortedUserGroups.validGroups[0];

            this.setState(
              {
                allUserGroupNamesUpper: allUserGroupNamesUpper,
                invalidUserGroups: sortedUserGroups.invalidGroups,
                userGroups: sortedUserGroups.validGroups,
              },
              this.setSelectedUserGroup(applySelectedUserGroup)
            );

            resolve(applySelectedUserGroup);
          });
        }
      ),
    ]);
  }

  setSelectedUserGroup(ug) {
    this.setState(
      {
        selectedUserGroup: ug,
        usersInSelectedGroup: [],
        invitesInSelectedGroup: [],
        expiredInvitesInSelectedGroup: [],
      },
      this.loadUserGroupData
    );
  }

  loadUserGroupData() {
    console.log(this.state.userGroupMode, this.state.selectedUserGroup);
    if (this.state.userGroupMode === "Users/Invites") {
      this.setState({
        loadingInvites: true,
        usersInSelectedGroup: [],
        invitesInSelectedGroup: [],
        expiredInvitesInSelectedGroup: [],
      });
      IafUserGroup.getUsers(this.state.selectedUserGroup, null, {
        _offset: 0,
        _pageSize: 200,
      }).then((users) => {
        users.sort((a, b) => {
          return a._lastname.localeCompare(b._lastname);
        });
        this.setState({ usersInSelectedGroup: users });
      });

      IafUserGroup.getInvites(this.state.selectedUserGroup).then((invites) => {
        let pendingInvites = invites._list.filter(
          (i) => i._status === "PENDING"
        );
        let expiredInvites = invites._list.filter(
          (i) => i._status === "EXPIRED"
        );

        this.setState({
          invitesInSelectedGroup: pendingInvites,
          expiredInvitesInSelectedGroup: expiredInvites,
          loadingInvites: false,
        });
      });
    }
  }

  toggleUserGroupEditable(e) {
    if (e) e.preventDefault();
    this.setState({
      userGroupNameEdit: this.state.selectedUserGroup._name,
      editingUserGroup: !this.state.editingUserGroup,
      userGroupNameEditError: null,
    });
  }

  onUserGroupNameChange = (e) => {
    const { value } = e.target;
    let error = null;
    let editedName = value;

    if (!value.trim()) {
      error = "UserGroup name is not allowed to be blank!";
    } else if (value.length > 50) {
      editedName = value.slice(0, 50);
    }

    this.setState({
      userGroupNameEdit: editedName,
      userGroupNameEditError: error,
    });
  };

  //currently we only update the usergroup's name
  updateUserGroup(e) {
    e.preventDefault();

    if (
      !this.state.allUserGroupNamesUpper.includes(
        this.state.userGroupNameEdit.toUpperCase()
      )
    ) {
      this.setState({ savingUserGroup: true });
      let updatedGroup = Object.assign({}, this.state.selectedUserGroup, {
        _name: this.state.userGroupNameEdit,
      });

      IafUserGroup.update(updatedGroup)
        .then((resGroup) => {
          this.toggleUserGroupEditable();
          this.setState({
            savingUserGroup: false,
            userGroups: [],
            invalidGroups: [],
            selectedUserGroup: null,
          });
          this.getAllUserGroups(resGroup);
        })
        .catch((err) => {
          console.error("error saving usergroup change", err);
          this.setState({ userGroupNameEditError: err.message });
        });
    } else {
      this.setState({
        userGroupNameEditError:
          "Duplicate UserGroup name! Please choose a different name.",
      });
    }
  }

  getAllUsers() {
    IafProj.getUsers(this.props.selectedItems.selectedProject).then(
      (allUsers) => {
        allUsers.sort((a, b) => {
          return a._lastname.localeCompare(b._lastname);
        });

        this.setState(
          { users: allUsers, selectedUser: allUsers[0] },
          this.loadUserData
        );
      }
    );
  }

  setSelectedUser(ug) {
    this.setState(
      {
        selectedUser: ug,
        userGroupsForSelectedUser: [],
        invitesForSelectedUser: [],
      },
      this.loadUserData
    );
  }

  async loadUserData() {
    this.setState({
      loadingInvites: true,
      userGroupsForSelectedUser: [],
      invitesForSelectedUser: [],
      expiredInvitesForSelectedUser: [],
    });

    let ugresults = await Promise.all(
      this.state.userGroups.map((ug) => {
        return IafUserGroup.getUsers(ug, null, {
          _offset: 0,
          _pageSize: 200,
        }).then((users) => {
          if (_.find(users, { _id: this.state.selectedUser._id })) return ug;
          else return null;
        });
      })
    );

    ugresults = ugresults.filter((u) => u);

    this.setState({ userGroupsForSelectedUser: ugresults });

    let invresults;
    //if currentUser is selected get the user's invites
    if (this.props.user._id === this.state.selectedUser._id) {
      invresults = await IafPassSvc.getUserInvites();
    } else {
      //we have to do a similar workaround for invites because there is no access
      //to fetch invites by another user so we have go usergroup by usergroup
      invresults = await Promise.all(
        this.state.userGroups.map((ug) => {
          return IafUserGroup.getInvites(ug).then((invs) => {
            return _.filter(invs._list, {
              _email: this.state.selectedUser._email,
            });
          });
        })
      );
    }

    invresults = _.flatten(invresults);
    invresults = invresults.filter((i) => i);
    let validInvites = invresults.filter((i) => i._status === "PENDING");
    let expiredInvites = invresults.filter((i) => i._status === "EXPIRED");

    this.setState({
      invitesForSelectedUser: validInvites,
      expiredInvitesForSelectedUser: expiredInvites,
      loadingInvites: false,
    });
  }

  async onCancelInvite(invite) {
    let result = await IafUserGroup.cancelInvite(invite._usergroup, invite._id);

    this.updateCurrentView();
  }

  async onResendInvite(invite) {
    try {
      // Cancel the original invite
      await IafUserGroup.cancelInvite(invite._usergroup, invite._id);

      const inviteData = [
        {
          _email: invite._email,
          _params: invite._params,
        },
      ];

      // Send a new invite
      await IafUserGroup.inviteUsersToGroup(invite._usergroup, inviteData);

      // Update the current view
      this.updateCurrentView();
    } catch (error) {
      // Handle any errors that occur during the process
      console.error("Error resending invite:", error);
    }
  }

  async onAcceptInvite(invite) {
    IafUserGroup.addUserToGroupByInvite(invite._usergroup, invite)
      .then((r) => {
        this._loadAsyncData();
      })
      .catch((e) => console.error("error accepting invite: ", e));
  }

  async canRemoveUser(user = false, group = false) {
    //if user is passed we are in usergroup mode
    //so we have a selectedUserGroup
    if (user) {
      if (
        this.state.selectedUserGroup._id ===
          this.props.selectedItems.selectedUserGroupId &&
        user._id === this.props.user._id
      )
        return {
          allow: false,
          message: "You can't remove your user from the current UserGroup",
        };
      else if (this.state.usersInSelectedGroup.length === 1)
        return {
          allow: false,
          message: "You can't remove the last User from a UserGroup",
        };
      else if (user._id === this.props.user._id)
        return {
          allow: true,
          message:
            "Are you sure you want to remove yourself from this UserGroup?",
        };
      else
        return {
          allow: true,
          message: "Confirm Remove User from Group",
        };
    } else if (group) {
      //otherwise we are in user mode
      //so we have a selectedUser
      if (
        group._id === this.props.selectedItems.selectedUserGroupId &&
        this.state.selectedUser._id === this.props.user._id
      )
        return {
          allow: false,
          message: "You can't remove your user from the current UserGroup",
        };
      else {
        let users = await IafUserGroup.getUsers(group, null, {
          _offset: 0,
          _pageSize: 200,
        });

        if (users.length === 1)
          return {
            allow: false,
            message: "You can't remove the last User from a UserGroup",
          };
        else if (this.state.selectedUser._id === this.props.user._id)
          return {
            allow: true,
            message:
              "Are you sure you want to remove yourself from this UserGroup?",
          };
        else
          return {
            allow: true,
            message: "Confirm Remove User from Group",
          };
      }
    }
  }

  async removeUser(user = false, group = false) {
    let removeUser, fromGroup;

    if (user) {
      removeUser = user;
      fromGroup = this.state.selectedUserGroup;
    } else {
      removeUser = this.state.selectedUser;
      fromGroup = group;
    }

    let canRemove;
    if (this.state.pageMode === "UserGroups")
      canRemove = await this.canRemoveUser(removeUser);
    else canRemove = await this.canRemoveUser(null, fromGroup);

    if (canRemove) {
      IafUserGroup.deleteUserFromGroup(fromGroup, removeUser).then((result) => {
        //if in usermode and we are removing the last usergroup for user
        //reload everything as the user will need to disappear from the user list
        if (
          this.state.pageMode === "Users" &&
          this.state.userGroupsForSelectedUser.length === 1
        )
          this._loadAsyncData();
        else this.updateCurrentView();
      });
    }
  }

  updateCurrentView() {
    if (this.state.pageMode === "UserGroups") {
      this.loadUserGroupData();
    } else {
      this.loadUserData();
    }
  }

  renderUserGroupsCard() {
    const { selectedUserGroup, userGroups, invalidUserGroups } = this.state;

    const isLoading = !selectedUserGroup;

    const renderThrobber = () => (
      <SimpleTextThrobber throbberText="Loading UserGroups" />
    );

    const renderGroupCards = (groups, additionalProps = {}) =>
      groups.map((group) => (
        <GroupCard
          key={group._id}
          group={group}
          {...additionalProps}
          isSelected={group._id === selectedUserGroup._id}
          onClick={(e) => this.setSelectedUserGroup(group)}
        />
      ));

    return (
      <div>
        {isLoading && renderThrobber()}

        <ul className="user-group-list">
          {renderGroupCards(userGroups, { selectable: true })}
        </ul>

        {invalidUserGroups.length > 0 && (
          <div className="other-groups">
            <span>Other Groups</span>
            <ul className="other-group-list">
              {renderGroupCards(invalidUserGroups, { disabled: true })}
            </ul>
          </div>
        )}
      </div>
    );
  }

  renderUserCard() {
    const { selectedUser, users } = this.state;
    const { user } = this.props;

    return (
      <div>
        {!selectedUser && <SimpleTextThrobber throbberText="Loading Users" />}
        <ul className="user-group-list">
          {users.map((u) => (
            <UserCard
              key={u._id}
              user={u}
              isCurrentUser={u._id === user._id}
              selectable={true}
              isSelected={selectedUser && u._id === selectedUser._id}
              onClick={() => this.setSelectedUser(u)}
            />
          ))}
        </ul>
      </div>
    );
  }

  renderUserGroupsModeView() {
    const {
      selectedUserGroup,
      editingUserGroup,
      userGroupMode,
      userGroupNameEdit,
      savingUserGroup,
      userGroupNameEditError,
      usersInSelectedGroup,
      invitesInSelectedGroup,
      expiredInvitesInSelectedGroup,
      loadingInvites,
      userGroupModes,
      users,
    } = this.state;
    const {
      allowUserGroupEdit,
      allowViewPermissions,
      allowManageUsers,
      allowManageInvites,
      allowManagePermissions,
    } = this.props.handler.config;
    const { user } = this.props;

    return (
      <div className="usergroup-mode-view">
        <div className="row1">
          {!editingUserGroup && (
            <div className="usergroup-info">
              <div className="usergroup-name">
                <h1>{selectedUserGroup ? selectedUserGroup._name : ""}</h1>
                {allowUserGroupEdit && selectedUserGroup && (
                  <span className="ug-btn">
                    <a href="#" onClick={this.toggleUserGroupEditable}>
                      edit
                    </a>
                  </span>
                )}
              </div>
              {allowViewPermissions && (
                <div>
                  <RadioButtons
                    options={userGroupModes}
                    value={userGroupMode}
                    onChange={this.onUserGroupModeChange}
                    labelPlacement="end"
                  />
                </div>
              )}
            </div>
          )}
          {editingUserGroup && (
            <div>
              <div className="usergroup-name editable">
                <h1>
                  <input
                    className="usergroup-name-input"
                    type="text"
                    disabled={savingUserGroup}
                    value={userGroupNameEdit}
                    onChange={this.onUserGroupNameChange}
                  />
                </h1>
                <span className="ug-btn">
                  {!savingUserGroup && !userGroupNameEditError && (
                    <a href="#" onClick={this.updateUserGroup}>
                      save
                    </a>
                  )}
                  {(savingUserGroup || userGroupNameEditError) && (
                    <span className="disabled-ug-btn">save</span>
                  )}
                </span>
                <span className="ug-btn">
                  {!savingUserGroup && (
                    <a href="#" onClick={this.toggleUserGroupEditable}>
                      cancel
                    </a>
                  )}
                  {savingUserGroup && (
                    <span className="disabled-ug-btn">cancel</span>
                  )}
                </span>
              </div>
              {userGroupNameEditError && (
                <div className="usergroup-name-error">
                  {userGroupNameEditError}
                </div>
              )}
            </div>
          )}
        </div>

        <hr />

        {userGroupMode === "Users/Invites" && (
          <div className="row2">
            <div className="usergroup-members">
              <div>
                <h3>Members</h3>
              </div>
              {usersInSelectedGroup.length === 0 && (
                <div className="throbber">
                  <SimpleTextThrobber throbberText="Loading UserGroup Members" />
                </div>
              )}
              <ul className="group-users-list">
                {usersInSelectedGroup.map((u) => (
                  <UserCard
                    key={u._id}
                    user={u}
                    isCurrentUser={u._id === user._id}
                    showActions={allowManageUsers}
                    canRemoveUser={this.canRemoveUser}
                    onRemoveUser={this.removeUser}
                  />
                ))}
              </ul>
            </div>
            <div className="usergroup-invites">
              <div>
                <h3>Invites</h3>
              </div>
              {loadingInvites && (
                <div className="throbber">
                  <SimpleTextThrobber throbberText="Loading UserGroup Invites" />
                </div>
              )}
              {!loadingInvites && invitesInSelectedGroup.length === 0 && (
                <span className="indent-header">No pending invites</span>
              )}
              <ul>
                {invitesInSelectedGroup.map((i) => (
                  <InviteCard
                    key={i._id}
                    invite={i}
                    isCurrentUser={i._email === user._email}
                    existingUser={_.find(users, {
                      _email: i._email,
                    })}
                    showActions={allowManageInvites}
                    onCancelInvite={this.onCancelInvite}
                    onResendInvite={this.onResendInvite}
                    onAcceptInvite={this.onAcceptInvite}
                  />
                ))}
              </ul>
              {expiredInvitesInSelectedGroup.length > 0 && (
                <div>
                  <span className="indent-header">Expired Invites</span>
                  <ul>
                    {expiredInvitesInSelectedGroup.map((i) => (
                      <InviteCard
                        key={i._id}
                        invite={i}
                        isCurrentUser={i._email === user._email}
                        existingUser={_.find(users, {
                          _email: i._email,
                        })}
                        showActions={allowManageInvites}
                        onCancelInvite={this.onCancelInvite}
                        onResendInvite={this.onResendInvite}
                        onAcceptInvite={this.onAcceptInvite}
                      />
                    ))}
                  </ul>
                </div>
              )}
            </div>
          </div>
        )}

        {userGroupMode === "Permissions" && (
          <div className="row2 table">
            <UserGroupPermissionTable
              usergroup={selectedUserGroup}
              allowManagePermissions={allowManagePermissions}
              itemFetchScript={handler.config.scripts?.itemFetchScript}
            />
          </div>
        )}
      </div>
    );
  }

  renderUserModeView() {
    const {
      selectedUser,
      userGroupsForSelectedUser,
      loadingInvites,
      invitesForSelectedUser,
      expiredInvitesForSelectedUser,
      users,
    } = this.state;

    const { allowManageInvites, allowManageUsers } = this.props.handler.config;

    const renderInviteCards = (invites) =>
      invites.map((invite) => (
        <InviteCard
          key={invite._id}
          invite={invite}
          isCurrentUser={invite._email === user._email}
          existingUser={_.find(users, { _email: invite._email })}
          showActions={allowManageInvites}
          onCancelInvite={this.onCancelInvite}
          onResendInvite={this.onResendInvite}
          onAcceptInvite={this.onAcceptInvite}
        />
      ));

    const renderThrobber = (throbberText) => (
      <div className="throbber">
        <SimpleTextThrobber throbberText={throbberText} />
      </div>
    );

    return (
      <div className="user-mode-view">
        <div className="row1">
          <div className="user-name-title">
            <h1>
              {selectedUser
                ? `${selectedUser._lastname}, ${selectedUser._firstname}`
                : ""}
            </h1>
          </div>
        </div>

        <hr />

        <div className="row2">
          <div className="member-usergroups">
            <div>
              <h3>UserGroups</h3>
            </div>
            {userGroupsForSelectedUser.length === 0 &&
              renderThrobber("Loading UserGroups")}
            <ul className="member-usergroup-list">
              {userGroupsForSelectedUser.map((group) => (
                <GroupCard
                  key={group._id}
                  group={group}
                  showActions={allowManageUsers}
                  canRemoveUser={this.canRemoveUser}
                  onRemoveUser={this.removeUser}
                />
              ))}
            </ul>
          </div>

          <div className="member-invites">
            <div>
              <h3>Invites</h3>
            </div>
            {loadingInvites && renderThrobber("Loading User Invites")}
            {!loadingInvites && invitesForSelectedUser.length === 0 && (
              <span className="indent-header">No pending invites</span>
            )}
            <ul>
              {renderInviteCards(invitesForSelectedUser)}
              {expiredInvitesForSelectedUser.length > 0 && (
                <div>
                  <span className="indent-header">Expired Invites</span>
                  <ul>{renderInviteCards(expiredInvitesForSelectedUser)}</ul>
                </div>
              )}
            </ul>
          </div>
        </div>
      </div>
    );
  }

  render() {
    const { user, selectedItems, handler } = this.props;
    const { allowUserGroupInvite, appUrl } = handler.config;
    const { pageMode, users, userGroups, pageModes } = this.state;
    const isUserGroupMode = pageMode === "UserGroups";

    return (
      <div className="user-group-view">
        {allowUserGroupInvite && (
          <StackableDrawer
            level={1}
            iconKey="fas fa-user-plus"
            isDrawerOpen={false}
          >
            <InviteForm
              appName={selectedItems.ipaConfig.appName}
              appUrl={appUrl}
              currentUser={user}
              users={users}
              userGroups={userGroups}
              project={selectedItems.selectedProject}
              onInvitesSent={this.updateCurrentView}
            />
          </StackableDrawer>
        )}

        <StackableDrawer
          level={allowUserGroupInvite ? 2 : 1}
          iconKey="fas fa-users"
          isDrawerOpen={true}
        >
          <div className="switchable-list-view">
            <div className="list-header">
              <div className="radio-btns">
                <RadioButtons
                  options={pageModes}
                  value={pageMode}
                  onChange={this.onModeChange}
                  labelPlacement="end"
                />
              </div>
            </div>
            <hr />
            {isUserGroupMode
              ? this.renderUserGroupsCard()
              : this.renderUserCard()}
          </div>
        </StackableDrawer>

        {isUserGroupMode
          ? this.renderUserGroupsModeView()
          : this.renderUserModeView()}
      </div>
    );
  }
}

export default UserGroupView;
