const filesPage = {
  //filesPage.js
  /**
   * @function  getFilesAndFolders
   * Gets files and folders for a given folder or based on a query - the files page hook calls this function
   * @return {Array}
   */
  async getFilesAndFolders (
    ctx, 
    folder, 
    filterType, 
    filterValue, 
    filtered,
    searchTerm, 
    global,
    backNav,
    pageSize, 
    pageNum,
    sortBy,
    sortDescend,
    libraries
    ) {
      const sortKeys = {
        'Name' : {
          'file': 'name',
          'folder': '_name'
        },
        'Updated' : {
          'file': 'versions.metadata._updatedAt',
          'folder': '_metadata._updatedAt'
        },
        'Created' : {
          'file': 'versions.metadata._createdAt',
          'folder': '_metadata._createdAt'
        },
        'Size' : {
          'file': 'versions.fileSize'
        }
      }
      const pageOptions = {
        "page": {
          "_pageSize": pageSize,
          "_offset": (pageNum - 1) * pageSize
        }
      }
      const fileOptions = {
        "sort": { [sortKeys[sortBy]['file']]: sortDescend ? -1 : 1 },
        ...pageOptions
      };
      const folderOptions = {
        "sort": { [sortKeys[sortBy]['folder']]: sortDescend ? -1 : 1 },
        ...pageOptions
      };
  
      let fetchedFolder = folder;
      let fetchedSubfolders = [];
      let fetchedSubfiles = [];
    try {
  
      if (filtered || global) {
        ({ fetchedSubfolders, fetchedSubfiles } = await this.getFoldersAndFilesByQuery(
          filterType, filterValue, searchTerm, global, ctx, folder, fileOptions, folderOptions, libraries
        ))
      } else {
        fetchedFolder = backNav ? folder : await this.getFolderByPath(folder._versions[0]._userAttributes.path, ctx, libraries);
        fetchedSubfolders = await this.loadFolderSubfolders(fetchedFolder, ctx, folderOptions, libraries);
        fetchedSubfiles = await this.loadFolderSubfiles(fetchedFolder, {}, ctx, fileOptions, libraries)
      } 
      if (fetchedFolder && fetchedSubfolders && fetchedSubfiles) {
        return ({
          fetchedFolder: fetchedFolder,
          fetchedSubfolders: fetchedSubfolders,
          fetchedSubfiles: fetchedSubfiles,
        })         
      }
    } catch (error) {
      console.error('getFilesAndFolders error:', error);
    }
  },

  /**
   * @function  getFoldersAndFilesByQuery
   * Gets files and folders based on your criteria and options query
   * @return {Array<User>}
   */
  async getFoldersAndFilesByQuery(
    filterType, 
    filterValue, 
    searchTerm, 
    global, 
    ctx, 
    currentFolder,
    fileOptions,
    folderOptions,
    libraries
  ) {
    const fileOnly = ['Size'];
    try {
      const filteredFolders = ( !fileOnly.includes(filterType) && filterValue !== 'File' ) ? 
        await this.getFoldersByQuery(
          (filterValue === 'Folder' ? '' : filterType), 
          (filterValue === 'Folder' ? '' : filterValue), 
          searchTerm, 
          global, 
          ctx, 
          currentFolder, 
          folderOptions,
          libraries 
        ) 
      : [];
      const filteredFiles = ( filterValue !== 'Folder' ) ? 
        await this.getFilesByQuery(
          (filterValue === 'File' ? '' : filterType), 
          (filterValue === 'File' ? '' : filterValue), 
          searchTerm, 
          global, 
          ctx, 
          currentFolder, 
          fileOptions,
          libraries
        ) 
      : [];
      return({
        fetchedSubfolders: filteredFolders,
        fetchedSubfiles: filteredFiles,
        fetchedFolder: currentFolder
      }) 
    } catch (error) {
      console.error('Error fetching getFoldersAndFilesByQuery:', error);
    }
  },

  /**
   * @function  getFoldersByQuery
   * Gets folders based on your criteria and options query
   * @return {Array<User>}
   */
  async getFoldersByQuery(
    filterType, 
    filterValue, 
    searchTerm, 
    global, 
    ctx, 
    currentFolder,
    options,
    libraries
  ) {
    const currentFolderPath = currentFolder._versions[0]._userAttributes.path;
    const filterObj = filterType.length > 0  && this.getFilterObj(filterType);
    console.log("filterObj", filterObj);
    const query = {
      ...(searchTerm.length > 0 && { "_name": { "$regex": searchTerm }}),
      ...(filterValue.toString().length > 0 && this.getFilterQuery(filterValue, filterObj.nestedProps, filterObj.format)),
      ...( global ? {"_userType": "file_container"} : {"_versions._userAttributes.parentPath": currentFolderPath} )
    };
    try {
      const queriedFolders = await this.getFolders( 
        query, 
        ctx,
        options,
        libraries
      );
      return queriedFolders
    } catch (error) {
      console.error('Error fetching getFoldersByQuery:', error);
    }
  },

  /**≠
   * @function  getFilesByQuery
   * Gets files based on your criteria and options query
   * @return {Array<String>}
   */
  async getFilesByQuery(
    filterType, 
    filterValue, 
    searchTerm, 
    global,
    ctx,
    currentFolder,
    options, 
    libraries
  ) {
    const { IafItemSvc } = libraries;
    try{
      const filterObj = filterType.length > 0 && this.getFilterObj(filterType);
      console.log("filterObj", filterObj);
      const searchQuery = {
        "name": { "$regex": searchTerm }
      };
      const query = {
        "query": {
          ...(searchTerm.length > 0 && searchQuery),
          ...(filterValue.toString().length > 0 && this.getFilterQuery(filterValue, filterObj.nestedProps, filterObj.format))
        }
      };
      const { page, sort } = options
      const projOptions = {
        "page": {
          "_pageSize": page._pageSize,
          "_offset": page._offset,
          "sort": sort
        }
      };
      console.log("projOptions", projOptions)
  
      const queryFileItems = ! global 
        ? 
          await IafItemSvc.getRelatedItems(
            currentFolder._id, 
            { ...query }, 
            ctx, 
            options
          )
        :
          await this.queryProjectFiles(
            { ...query }, 
            ctx,
            projOptions,
            libraries
          ) 
      ; 
  
      const responseItems = ! global ? queryFileItems._list : queryFileItems;
      console.log("responseItems", responseItems);
  
      return responseItems.length > 0 ? responseItems : []
      
       
    } catch (error) {
      console.error('Error fetching getFilesByQuery:', error);
    }
  },
  //folderView.js
  /**
   * @function  getWorkspaceUsers
   * Gets the project's workspace users
   * @return {Array<User>}
   */
  async getWorkspaceUsers(ctx, libraries) {
    const { IafWorkspace } = libraries;
    try {
      console.log("getting workspace");
      const workspaces = await IafWorkspace.getAll(ctx);
      console.log("getting workspace users"); 
      const workspaceUsers = await IafWorkspace.getUsers(workspaces[0], undefined, ctx);
      console.log("workspace users", workspaceUsers ); 
      return workspaceUsers
    } catch (error) {
      console.error('Error getting workspace users:', error);
    }
  },

  /**
   * @function  getContext
   * Gets the project and user's context
   * @return {Ctx}
   */
  async getContext(libraries) {
    console.log("getContext libraries", libraries);
    try {
      console.log("getting context");
      const { ctx } = await this.getCurrentProjectInfo(libraries);
      console.log("context response", ctx);
      return ctx
    } catch (error) {
      console.error('Error getting context:', error);
    }
  },

  /**
   * @function  getCurrentProjectInfo
   * Get the current project information
   * @return {project, Ctx}
   */
  async getCurrentProjectInfo(libraries) {
    console.log("getCurrentProjectInfo libraries", libraries);
    const { IafProj, IafSession } = libraries;
    try {
      let project = await IafProj.getCurrent();
      console.log("project response", project);
      let ctx = { _namespaces: project._namespaces };
      ctx.authToken = await IafSession.getAuthToken();
      let token = ctx.authToken;
      let _namespaces = ctx._namespaces;
      let configUrl = await IafSession.getConfig();

      return { project, ctx, token, configUrl, _namespaces };
    } catch (err) {
      return err;
    }
  },

  /**
   * @function  getFolders
   * Gets folders by the query you pass
   * @return {Array<NamedFileCollection>}
   */
  async getFolders( nestedQuery, ctx, options, libraries) {
    const { IafItemSvc } = libraries;
    console.log("libraries", libraries)
    console.log("IafItemSvc", IafItemSvc)
    try {
      console.log("getFolders running");
      const criteria = {
        query: {
          "_itemClass": "NamedFileCollection",
          ...nestedQuery
        }
      };
      console.log("getFolders options", options);
      const folderArray = await IafItemSvc.getNamedUserItems(criteria, ctx, options);
      console.log("folderArray", folderArray);
      return folderArray._list
    } catch (error) {
      console.error('Error fetching folder:', error);
    }
  },

  /**
   * @function  getFolderByPath
   * Gets a NamedFileCollection when you pass its path
   * @return {NamedFileCollection}
   */
  async getFolderByPath(path, ctx, libraries) {
    try {
      console.log("getFolderByPath running");
      const nestedQuery = {
        "_versions._userAttributes.path": path
      };
      const folderArray = await this.getFolders(nestedQuery, ctx, undefined, libraries);
      return folderArray && folderArray[0]
    } catch (error) {
      console.error('Error fetching folder:', error);
    }
  },

  /**
   * @function loadFolderSubfolders
   * Loads a folder's subfolders
   * @return {Array<NamedFileCollection>}
   */
  async loadFolderSubfolders(folder, ctx, options, libraries) {
    try {
      if (folder) {
        console.log("subFolderNfcs running");
        const nestedQuery = {
          "_versions._userAttributes.parentPath": folder._versions[0]._userAttributes.path
        };
        const subfolders = await this.getFolders(nestedQuery, ctx, options, libraries);
        console.log("loadFolderSubfolders result: ", subfolders);
        return subfolders;
      } 
    } catch (error) {
      console.error('Error fetching current folder child folders:', error);
    }
  },

  /**
   * @function loadFolderSubfiles
   * Loads a folder's subfileså
   * @return {Array<FileItem>}
   */
  async loadFolderSubfiles(folder, query, ctx, options, libraries) {
    console.log("loadFolderSubfiles", folder, query, ctx, options, libraries);
    const { IafFile } = libraries;
    if (folder) {
      try {
        console.log("loadFolderSubfiles running");
        const relatedFileItems = await IafFile.getFileItems(folder, query, ctx, options);
        console.log("relatedFileItems", relatedFileItems);
        return (relatedFileItems && relatedFileItems._list) ? relatedFileItems._list : [];
      } catch (error) {
        console.error('Error fetching folder child files:', error);
      }
    } 
  },

  /**
   * @function queryProjectFiles
   * Loads a folder's subfiles by query
   * @return {Array<FileItem>}
   */
  async queryProjectFiles( nestedQuery, ctx, options, libraries ) {
    console.log("queryProjectFiles options", options);
    console.log("libraries", libraries);
    const { IafFetch } = libraries;
    const namespace = ctx._namespaces[0];
    const pageSize = options.page._pageSize;
    const offset = options.page._offset;
    const sort = JSON.stringify(options.page.sort);
    const query = JSON.stringify(nestedQuery.query);

    let url = `${endPointConfig.objectModelServiceOrigin}/omapi/${namespace}/projectfileitems?_pageSize=${pageSize}&_offset=${offset}&sort=${sort}&query=${query}`

    console.log('URL', url);

    try {
      const response = await IafFetch.doGet(url, ctx, options);
      console.log("queryProjectFiles fetch response", response);
      const matches = await response._result.matches;
    
      console.log('matches', response);
      return matches;
    } catch (error) {
      console.log("queryProjectFiles error", error);
    }
  },



  /**
   * @function  getFileAndFolderValueOptions
   * Gets filter options based on the global or local folders
   * @return {Array<String>}
   */
  async getFileAndFolderValueOptions(type, global, ctx, currentFolder, users, libraries) {
    const fileOnly = ['Size'];
    const fileTypes = [{ dbValue: 'Folder', formatted: 'Folder'}, { dbValue: 'File', formatted: 'File'}];
    let distinctItems = [];
    if (type !== 'File type') {
      try {
        const folderValues = !fileOnly.includes(type) ? await this.getFolderFilterOptions(
          type, 
          global,
          ctx,
          currentFolder,
          users,
          libraries 
        ) : [];
        console.log("folderValues", folderValues);
        const filesValues = await this.getFileFilterOptions(
          type, 
          global,
          ctx,
          currentFolder,
          users,
          libraries
        );
        console.log("filesValues", filesValues); 
        const items = [...folderValues, ...filesValues];
        distinctItems = items.filter((obj, index, self) => 
          index === self.findIndex((t) => (
            t.formatted === obj.formatted
          ))
        );
      } catch (error) {
        console.error('Error fetching getFileAndFolderValueOptions:', error);
      }
    } else {
      distinctItems = fileTypes;
    }
    return distinctItems
  },

  /**
   * @function  getFilteredFilesAndFolders
   * Gets filter options based on the global or local folders
   * @return {Array<String>}
   */
  async getFilteredFilesAndFolders(type, value, global, ctx, currentFolder ) {
    const fileOnly = ['Size'];
    try {
      const filteredFolders = ( !fileOnly.includes(type) && value !== 'File' ) ? await this.getFilteredFolders(
        type, value, global, ctx, currentFolder._versions[0]._userAttributes.path 
      ) : [];
      const filteredFiles = ( value !== 'Folder' ) ? await this.getFilteredFiles(
        type, value, global, ctx, currentFolder
      ) : [];
      return({
        folders: filteredFolders,
        files: filteredFiles
      }) 
    } catch (error) {
      console.error('Error fetching getFilteredFilesAndFolders:', error);
    }
  },

  /**
   * @function  createFolder
   * Creates a folder
   * @return {NamedFileCollection}
   */
  async createFolder(folderDef, currentFolder, ctx, libraries) {
    const { IafFile } = libraries;
    console.log("creating new folder:", folderDef._name);
    try {
      return await IafFile.createContainer(currentFolder, folderDef, ctx);
    } catch (error) {
      console.error('Error getting container:', error);
    }
  },

  /**
   * @function  updateFileItem
   * Updates a file
   * @return {}
   */
  async updateFileItem(container, file, ctx, libraries) {
    const { IafFile } = libraries;
    console.log("updating file:", file);
    try {
      const updatedFile = await IafFile.updateFileItem(
        container, 
        file, 
        ctx
      );
      console.log("updatedFile", updatedFile);
      return updatedFile
    } catch (error) {
      console.error('Error updating file:', error);
    }
  },

  /**
   * @function  updateFileCollection
   * Updates a file collection
   * @return {}
   */
  async updateFileColl(folder, ctx, libraries) {
    const { IafItemSvc } = libraries;
    console.log("updating folder:", folder);
    try {
      const updatedColl = await IafItemSvc.updateNamedUserItem(
        folder, 
        ctx
      );
      console.log("updatedColl", updatedColl);
      return updatedColl
    } catch (error) {
      console.error('Error updating folder:', error);
    }
  },

  /**
   * @function deleteFile
   * Deletes a file item and its source file 
   */
  async deleteFile(fileItem, folder, ctx, libraries) {
    const { IafFileSvc, IafFile } = libraries;
    try {
      let deletedSrc = false;
      await IafFileSvc.deleteFile(fileItem._fileId, ctx)
        .then(response => {
          if (response === 'ok: 204') {
            deletedSrc = true;
          }
        })
        .catch(error => {
          console.error('Error deleting source file:', error);
        });
      if (deletedSrc) {
        await IafFile.deleteFileItem(folder, fileItem, ctx)
          .then(response => {
            if (response == 'ok: 204') {
              return true
            }
          })
          .catch(error => {
            console.error('Error deleting file item:', error);
          });
      }
    } catch (error) {
      console.error('Error deleting file:', error);
    }
  },

  /**
   * @function deleteFileVersion
   * Deletes a file item version and its source file version
   */
  async deleteFileVersion(fileItem, fileItemVersion, folder, ctx, libraries) {
    const { IafFileSvc, IafFile } = libraries;
    try {
      let deletedSrc = false;
      
      await IafFileSvc.deleteFileVersion(fileItem._fileId, fileItemVersion._id, ctx)
        .then(response => {
          if (response === 'ok: 204') {
            deletedSrc = true;
          } else {
            console.log("deleteFileVersion response", response)
          }
        })
        .catch(error => {
          console.error('Error deleting source file:', error);
        });
      if (deletedSrc) {
        const options = {
          userItemVersionId: fileItemVersion._id
        }
        await IafFile.deleteFileVersion(folder, fileItem, ctx)
          .then(response => {
            if (response == 'ok: 204') {
              return true
            }
          })
          .catch(error => {
            console.error('Error deleting file item:', error);
          });
      }
    } catch (error) {
      console.error('Error deleting file:', error);
    }
  },

  /**
   * @function deleteSubfiles
   * Deletes a folder's subfiles
   * @return {Array<"ok: 204">}
   */
  async deleteSubfiles(subfiles, folder, ctx, libraries) {
    const deletePromises = [];
    subfiles.forEach(subfile => {
      deletePromises.push(this.deleteFile(subfile, folder, ctx, libraries));
    });

    Promise.all(deletePromises)
    .then(responses => {
      const allSuccess = true;
      responses.forEach(response => {
        if (response !== true) {allSuccess = false}
      });
      return allSuccess
    })
    .catch(error => {
      console.error('Error deleting subfiles:', error);
    });
  },

  /**
   * @function deleteFolder
   * Deletes a folder
   * @return {"ok: 204"}
   */
  async deleteFolder(folder, ctx, libraries) {
    const { IafItemSvc } = libraries;
    await IafItemSvc.deleteNamedUserItem(folder._id, ctx)
      .then(response => {
        if (response === 'ok: 204') {
          return true
        }
      })
      .catch(error => {
        console.error('Error deleting folder:', error);
      });
  },

  /**
   * @function deleteSubfolders
   * Deletes a folder
   * @return {String<"ok: 204">}
   */
  async deleteSubfolders(subfolders, ctx, libraries) {
    const deletePromises = [];
    subfolders.forEach(subfolder => {
      deletePromises.push(this.deleteFolder(subfolder, ctx, libraries));
    });

    Promise.all(deletePromises)
    .then(responses => {
      const allSuccess = true;
      responses.forEach(response => {
        if (response !== true) {allSuccess = false}
      });
      return allSuccess
    })
    .catch(error => {
      console.error('Error deleting subfolders:', error);
    });
  },

  //folderFilter.js
  /**
   * @function  getFolderFilterOptions
   * Gets filter options based on the global or local folders
   * @return {Array<String>}
   */
  async getFolderFilterOptions(type, global, ctx, currentFolder, users, libraries) {
    console.log("running getFolderFilterOptions");
    const filterObject = this.getFilterObj(type);
    console.log("filterObject getFolderFilterOptions", filterObject);
    let localNestedQuery = {
      "_versions._userAttributes.parentPath": currentFolder._versions[0]._userAttributes.path
    };
    try {
      let filterOptions = this.getFilterOptions(
        global ? await this.getFolders(undefined, ctx, undefined, libraries) : await this.getFolders(localNestedQuery, ctx, undefined, libraries),
        filterObject.nestedProps,
        filterObject.format,
        users
      );
      console.log("getFolderFilterOptions", filterOptions)
      return filterOptions
    } catch (error) {
      console.error('Error fetching getFolderFilterOptions:', error);
    }
  },

  formatDate(timestamp) {
    const date = new Date(timestamp);
    const day = date.getDate();
    const month = date.getMonth() + 1; 
    const year = date.getFullYear();
    const hours = date.getHours();
    const minutes = date.getMinutes();
  
    const doubleDigitHours = hours < 10 ? `0${hours}` : hours;
    const doubleDigitMins = minutes < 10 ? `0${minutes}` : minutes;
  
    return(`${day}-${month}-${year}, ${doubleDigitHours}:${doubleDigitMins} GMT`)
  },

  //folderView.js
  /**
   * @function  getFileFilterOptions
   * Gets the filter value option for a given filter property
   * @return {Array<String>}
   */
    async getFileFilterOptions(
    type,
    global,
    ctx,
    folder,
    users,
    libraries
  ) {
  const filterObject = this.getFilterObj(type);
  const concatProp = this.concatProps(filterObject.nestedProps); 
    try {
      let queryFileItems = [];
      console.log("getFileFilterOptions running");
      if (type !== 'Size') {
        queryFileItems = global ? 
          await this.queryProjectFiles({}, ctx, {
            "page": {
              "_pageSize": 100,
              "_offset": 0,
              "sort": { [concatProp]: 1 }
            }    
          }, libraries) 
        : await this.loadFolderSubfiles(folder, {}, ctx, {
              "project": { [concatProp]: 1 },
              "sort": { [concatProp]: 1 },
            }, 
            libraries
          );
          
      } else {
        queryFileItems = global ? 
          await this.queryProjectFiles({}, ctx, {
            "page": {
              "_pageSize": 100,
              "_offset": 0,
              "sort": { [concatProp]: 1 }
            }
          }, libraries)
        : await this.loadFolderSubfiles(folder, {}, ctx, {}, libraries);
      }
      console.log("queryFileItems", queryFileItems);
      const filterOptions = this.getFilterOptions(
        queryFileItems, 
        filterObject.nestedProps,
        filterObject.format,
        users
      );
      console.log("filterOptions", filterOptions);
      return filterOptions
    } catch (error) {
      console.error('Error fetching getFileFilterOptions:', error);
    }
  },
  
  //sortKeys.js
  sortKeys: {
    'Name' : {
      'file': 'name',
      'folder': '_name'
    },
    'Updated' : {
      'file': 'versions.metadata._updatedAt',
      'folder': '_metadata._updatedAt'
    },
    'Created' : {
      'file': 'versions.metadata._createdAt',
      'folder': '_metadata._createdAt'
    },
    'Size' : {
      'file': 'versions.fileSize'
    }
  },
  

  //filterObjs.js

  filterObjs: {
    'Created date': {
      nestedProps: ['_metadata', '_createdAt'],
      format: 'date',
    },
    'Updated date': {
      nestedProps: ['_metadata', '_updatedAt'],
      format: 'date',
    },
    'Updater': {
      nestedProps: ['_metadata', '_updatedById'],
      format: 'name',
    },
    'Creator': {
      nestedProps: ['_metadata', '_createdById'],
      format: 'name',
    },
    'Size': {
      nestedProps: ['versions', '0', 'fileSize'],
      format: 'size',
    },
    'File type': {

    }
  },

  getFilterObj( filterType ) {
    return this.filterObjs[filterType]
  },

  getFilterObjs( ) {
    return this.filterObjs
  },
  
  //options router
  
  getFilterOptions(objects, nestedProps, format, users) {
    console.log("getting getFilterOptions", objects);
    let filterOpts = [];
    objects.forEach((obj) => {
      const val = this.extractValueFromObject(obj, nestedProps);
      console.log("getFilterOptions extracted val", val)
      filterOpts.push(
        {
          dbValue: val,
          formatted: this.formatDbVal(format, val, users)
        } 
      );
    });
    console.log("filterOpts", filterOpts);
    return filterOpts
  },
  
  //format router

  formatDbVal(format, val, users) {
    console.log("running formatDbVal");
    switch (format) {
      case 'date':
        return this.formatDateDMY(val);
      case 'name': 
        return this.getUpdatedByUser(val, users);
      case 'size':
        return this.formatSize(val)
      default:
        return 'formatDbVal type not known'
    }
  },
  
  //format functions
  
  formatDateDMY(timestamp) {
    const date = new Date(timestamp);
    const day = date.getDate();
    const month = date.getMonth() + 1; 
    const year = date.getFullYear();
  
    return(`${day}-${month}-${year}`)
  },
  
  formatSize(val) { return `${val} bytes`},
  
  getUpdatedByUser (userId, users) {
    console.log("getUpdatedByUser userId", userId);
    console.log("getUpdatedByUser users", users);
    const user = users.filter(user => user._id === userId);
    if (user[0]) {
      return user[0]._firstname + " " + user[0]._lastname
    }
  },
  
  concatProps(propertiesArray) {
    if (propertiesArray.length === 1) {
      return propertiesArray[0];
    }
    return propertiesArray.join('.');
  },
  
  //value extractor
  
  extractValueFromObject(object, propertiesArray) {
    console.log("extractValueFromObject objects", object);
    console.log("extractValueFromObject propertiesArray", propertiesArray);
    let obj = object;
    for (const property of propertiesArray) {
      if (obj && obj.hasOwnProperty(property)) {
        obj = obj[property];
      } else {
        return undefined
      }
    }
    console.log("obj", obj);
    return obj;
  },
  
  // query routers

  getFilterQuery(val, nestedProps, format) {
    console.log("running getFilterQuery");
    switch (format) {
      case 'date':
        return this.onDateQuery(val, nestedProps);
      case 'name': 
        return this.userQuery(val, nestedProps);
      case 'size':
        return this.sizeQuery(val, nestedProps)
      default:
        return 'Type not known'
    }
  },
  
  // queries
  
  onDateQuery(ts, nestedProps) {
    const date = new Date(ts);
    const dateStart = new Date(date);
    const dateEnd = new Date(date);
    dateStart.setHours(0, 0, 0, 0);
    dateEnd.setHours(23, 59, 59, 999);
    const queryProp = this.concatProps(nestedProps);
  
    const query = {
      "$and": [
        {
          [queryProp]: { "$gte": dateStart.getTime() }
        }, 
        {
          [queryProp]: { "$lte" : dateEnd.getTime() }
        }
      ] 
    };
    
    return query
  },
  
  userQuery(userId, nestedProps) {
    const queryProp = this.concatProps(nestedProps);
    const query = {
      [queryProp]: userId
    };
    return query
  },
  
  sizeQuery(val, nestedProps) {
    const queryProp = this.concatProps(nestedProps);
    const query = {
      [queryProp]: val
    };
    return query
  }
};
export default filesPage;