class InputValidation {
  constructor(params, libraries, ctx) {
    this.params = params;
    this.libraries = libraries;
    this.ctx = ctx;
  }

  async validate() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `RefApp Executing validate method, orchrun_id: ${this.params.orchRunId}`
        );
        this.#validateFileDetails();

        if (await this.#modelExists()) {
          console.log(
            `RefApp Model Imported already, Please remove existing import and try again, orch_run_id: ${this.params.orchRunId}`
          );

          throw new Error(
            "RefApp Model Imported already, Please remove existing import and try again"
          );
        }

        console.log(
          `RefApp Basic input validation completed, orchrun_id: ${this.params.orchRunId}`
        );

        resolve(true);
      } catch (error) {
        console.log(
          `RefApp Error at validate method, orchrun_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  #validateFileDetails() {
    if (!this.params.actualParams._fileId)
      throw new Error("RefApp Invalid Parameters - _fileId is Invalid");
    if (!this.params.actualParams._fileVersionId)
      throw new Error("RefApp Invalid Parameters - _fileVersionId is Invalid");
    if (this.params.ext != "bimpk" && this.params.ext != "sgpk")
      throw new Error("RefApp Invalid model file extension");
  }

  async #modelExists() {
    return new Promise(async (resolve, reject) => {
      try {
        const { PlatformApi } = this.libraries;
        const fileIdAttribute = `_versions._userAttributes.${this.params.ext}.fileId`;
        const fileVersionIdAttribute = `_versions._userAttributes.${this.params.ext}.fileVersionId`;
        const criteria = {
          query: {
            [fileIdAttribute]: this.params.actualParams._fileId,
            [fileVersionIdAttribute]: this.params.actualParams._fileVersionId,
            "_versions.all": true,
          },
        };
        const project = { _namespaces: this.ctx._namespaces };

        const models = await PlatformApi.IafProj.getModels(
          project,
          criteria,
          this.ctx
        );
        const doesExist = models ? models.length > 0 : false;

        resolve(doesExist);
      } catch (error) {
        console.log(
          `RefApp Error at modelExists method, orchrun_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }
}

class BimpkImport {
  #assetTypeMap = null;
  #propMap;
  #typeMap;

  constructor(params, libraries, ctx) {
    this.params = params;
    this.libraries = libraries;
    this.ctx = ctx;
  }

  async initialize() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `RefApp Bimpk import initialized, orch_run_id: ${this.params.orchRunId}`
        );
        const { PlatformApi, IafScriptEngine, ModelFileReader } =
          this.libraries;
        const { IafItemSvc } = PlatformApi;
        let param = {
          ...this.params.actualParams,
          filename: this.params.filename,
          ext: this.params.ext,
          orchRunId: this.params.orchRunId,
        };

        // set global variables first
        await Promise.all([
          IafScriptEngine.setVar("namespaces", this.ctx._namespaces),
          IafScriptEngine.setVar("package_name", param.filename),
          IafScriptEngine.setVar(
            "package_name_short",
            param.filename.substring(0, 11)
          ),
          IafScriptEngine.setVar("bimpk_fileid", param._fileId),
          IafScriptEngine.setVar("bimpk_fileVersionId", param._fileVersionId),
        ]);

        const { _list = [] } =
          (await IafItemSvc.getNamedUserItems(
            {
              query: {
                _userType: "bim_model_version",
                "_versions._userAttributes.bimpk.fileId": param._fileId,
                _itemClass: "NamedCompositeItem",
              },
            },
            this.ctx,
            {}
          )) || {};
        const bim_model = _list[0];

        if (bim_model) {
          console.log("BimpkImport init bim_model: ", JSON.stringify(bim_model))
          await IafScriptEngine.setVar("bim_model", bim_model);
        } else {
          console.log("bim model not found: ", bim_model)
        }

        if (bim_model) {
          //
          console.time(`${this.params.orchRunId}: RefApp createBIMCollectionVersion`);
          await this.#createBIMCollectionVersion();
          console.timeEnd(
            `${this.params.orchRunId}: RefApp createBIMCollectionVersion`
          );
        } else {
          console.time(`${this.params.orchRunId}: RefApp createBIMCollections`);
          await this.#createBIMCollections();
          console.timeEnd(`${this.params.orchRunId}: RefApp createBIMCollections`);
        }

        const [
          bim_model_nci,
          model_els_coll,
          model_els_props_coll,
          model_type_el_coll,
          data_cache_coll,
          model_geom_file_coll,
          model_geom_views_coll,
        ] = await Promise.all([
          IafScriptEngine.getVar("bim_model"),
          IafScriptEngine.getVar("model_els_coll"),
          IafScriptEngine.getVar("model_els_props_coll"),
          IafScriptEngine.getVar("model_type_el_coll"),
          IafScriptEngine.getVar("data_cache_coll"),
          IafScriptEngine.getVar("model_geom_file_coll"),
          IafScriptEngine.getVar("model_geom_views_coll"),
        ]);
        let result = {
          bim_model_nci: bim_model_nci,
          model_els_coll: model_els_coll,
          model_els_props_coll: model_els_props_coll,
          model_type_el_coll: model_type_el_coll,
          data_cache_coll: data_cache_coll,
          model_geom_file_coll: model_geom_file_coll,
          model_geom_views_coll: model_geom_views_coll,
        }
        console.log("BimpkImport Model vars result: ", JSON.stringify(result));

        const bimModelId = bim_model_nci._id;
        const relatedColls = await IafScriptEngine.addRelatedCollections(
          {
            namedCompositeItemId: bimModelId,
            relatedCollections: [
              model_els_coll._userItemId,
              model_els_props_coll._userItemId,
              model_type_el_coll._userItemId,
              data_cache_coll._userItemId,
              model_geom_file_coll._userItemId,
              model_geom_views_coll._userItemId,
            ],
          },
          ctx
        );
        console.log('BimpkImport relatedColls: ', JSON.stringify(relatedColls))

        const myCols = {
          model_els_coll,
          model_els_props_coll,
          model_type_el_coll,
          data_cache_coll,
          model_geom_file_coll,
          model_geom_views_coll,
        };

        try {
          this.#assetTypeMap = await IafScriptEngine.getItems(
            {
              collectionDesc: {
                _userType: "iaf_dt_type_map_defs_coll",
                _namespaces: this.ctx._namespaces,
              },
              options: {
                page: { getAllItems: true },
              },
            },
            this.ctx
          );
          console.log('BimpkImport this.#assetTypeMap: ', JSON.stringify(this.#assetTypeMap));
          console.log(
            `RefApp Asset map type received, orch_run_id: ${this.params.orchRunId} `,
            typeof this.#assetTypeMap
          );
        } catch (err) {
          console.log(
            `RefApp Type Map collection does not exist, orch_run_id: ${this.params.orchRunId}`
          );
        }

        //download zipped model file
        const zipModelFile = await ModelFileReader.downloadAndUnzipModelFile(param, ctx);
        console.log('zipModelFile: ', JSON.stringify(zipModelFile))
        const { bimFilePath, manifest } = zipModelFile;
        const { files, occurrences } = manifest;
        const fileLength = files.length - 1;

        let index = -1;
        for (const model of files) {
          console.log(`BimpkImport Working on model file: `, JSON.stringify(model));
          index += 1;
          console.log(`RefApp Occurrence file ${index}, orch_run_id: ${this.params.orchRunId}`);
          // find occurances
          let occs = occurrences.filter((d) => d.source === model.id);
          console.log('RefApp BimpkImport: Occurrences occs: ', JSON.stringify(occs));
          if (!occs?.length) {
            console.warn(`RefApp Warning: Occurrences missing for model: ${model.id}, orch_run_id: ${this.params.orchRunId}`);
            console.log(`RefApp Warning: Occurrences missing for model: ${model.id}, orch_run_id: ${this.params.orchRunId}`);
            occs = [];
          }

          for (const occ of occs) {
            const filePath = bimFilePath + "/" + occ.data.objects;
            let newOccurrence = true;
            let letsLoop = true;

            while (letsLoop) {
              console.log(`RefApp Requesting bimpk batchlet, orch_run_id: ${this.params.orchRunId} for occ ${JSON.stringify(occ)}`);
              console.time(`${this.params.orchRunId}: getModelBatchlet`);
              const { bimBatch, endOfFile } =
                await ModelFileReader.getModelBatchlet(
                  filePath,
                  this.params.orchRunId
                );
              console.log(`RefApp BimpkImport Bim batchlet received: ${bimBatch?.objects?.length}, orch_run_id: ${this.params.orchRunId}, occ ${JSON.stringify(occ)}`);
              console.log(`RefApp BimpkImport Bim batchlet: ${JSON.stringify(bimBatch)}, orch_run_id: ${this.params.orchRunId}, occ ${JSON.stringify(occ)}`);
              console.timeEnd(`${this.params.orchRunId}: RefApp getModelBatchlet`);

              console.time(`${this.params.orchRunId}: RefApp extractBimpk`);
              await this.#extractBimpk(model.name, bimBatch, newOccurrence);
              
              console.time(`${this.params.orchRunId}: RefApp createRelatedItemsAndRelationships`);
              await this.#createRelatedItemsAndRelationships(
                newOccurrence,
                myCols
              );
              console.timeEnd(
                `${this.params.orchRunId}: RefApp createRelatedItemsAndRelationships`
              );

              newOccurrence = false;

              if (endOfFile) break;
            }
          }

          if (fileLength === index) {
            const result = {
              filecolid: myCols.model_geom_file_coll._userItemId,
              viewcolid: myCols.model_geom_views_coll._userItemId,
              compositeitemid: bimModelId,
              bim_model_nci: bim_model_nci,
              myCollections: myCols,
              filename: param.filename,
              _fileId: param._fileId,
              _fileVersionId: param._fileVersionId,
              bimFilePath,
              manifest,
            };

            await IafScriptEngine.clearVars();
            console.log('Bimpk import res: ', JSON.stringify(result))
            return resolve(result);
          }
        }
      } catch (error) {
        reject(error);
      }
    });
  }

  async #createBIMCollectionVersion() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `RefApp Found Previous Model Creating Versions, orch_run_id: ${this.params.orchRunId}`
        );
        const { IafScriptEngine } = this.libraries;
        const [
          bimModel,
          bimpkFileId,
          bimpkFileVersionId,
          packagename,
          packagenameShort,
        ] = await Promise.all([
          IafScriptEngine.getVar("bim_model"),
          IafScriptEngine.getVar("bimpk_fileid"),
          IafScriptEngine.getVar("bimpk_fileVersionId"),
          IafScriptEngine.getVar("package_name"),
          IafScriptEngine.getVar("package_name_short"),
        ]);
        const newModelVer = {
          namedUserItemId: bimModel._id,
          _userAttributes: {
            bimpk: {
              fileId: bimpkFileId,
              fileVersionId: bimpkFileVersionId,
            },
          },
        };
        const modelRelatedCollection = await IafScriptEngine.getCollectionsInComposite(
          bimModel._id,
          null,
          this.ctx
        )

        //IafScriptEngine.createNamedUserItemVersion(newModelVer, this.ctx),
        console.log(`RefApp modelRelatedCollection, orch_run_id: ${this.params.orchRunId}`, JSON.stringify(modelRelatedCollection));
        console.log(`RefApp Created BIM Collection Version bim_model version, orch_run_id: ${this.params.orchRunId}`);

        let model_els_coll,
          model_els_props_coll,
          model_type_el_coll,
          model_geom_file_coll,
          model_geom_views_coll,
          data_cache_coll;

        if (modelRelatedCollection.length) {

          for (const obj of modelRelatedCollection) {
            if (obj._userType === "rvt_elements" && !model_els_coll) {
              model_els_coll = obj;
            } else if (
              obj._userType === "rvt_element_props" &&
              !model_els_props_coll
            ) {
              model_els_props_coll = obj;
            } else if (
              obj._userType === "rvt_type_elements" &&
              !model_type_el_coll
            ) {
              model_type_el_coll = obj;
            } else if (
              obj._userType === "bim_model_geomresources" &&
              !model_geom_file_coll
            ) {
              model_geom_file_coll = obj;
            } else if (
              obj._userType === "bim_model_geomviews" &&
              !model_geom_views_coll
            ) {
              model_geom_views_coll = obj;
            } else if (obj._userType === "data_cache" && !data_cache_coll) {
              data_cache_coll = obj;
            }
          }

          if (!data_cache_coll) {
            const data_cache_coll_def = {
              _name: packagename + "_data_cache",
              _shortName: packagenameShort + "_data_cache",
              _description: "Data cached about imported model",
              _userType: "data_cache",
              _namespaces: this.ctx._namespaces,
            };

            data_cache_coll = await IafScriptEngine.createCollection(
              data_cache_coll_def,
              this.ctx
            );

            console.log(
              `RefApp Created Model Data Cache, orch_run_id: ${this.params.orchRunId}`
            );
          }

          console.log(
            `RefApp Created Model versions, orch_run_id: ${this.params.orchRunId}`
          );
          // create the versions
          await Promise.all([
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_els_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_els_props_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_type_el_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: data_cache_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_geom_file_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_geom_views_coll._userItemId },
              this.ctx
            )
          ]);

          // set them in global variables

          await IafScriptEngine.setVar("model_els_coll", model_els_coll)
          await IafScriptEngine.setVar("model_els_props_coll", model_els_props_coll)
          await IafScriptEngine.setVar("model_type_el_coll", model_type_el_coll)
          await IafScriptEngine.setVar("data_cache_coll", data_cache_coll)
          await IafScriptEngine.setVar("model_geom_file_coll", model_geom_file_coll)
          await IafScriptEngine.setVar("model_geom_views_coll", model_geom_views_coll)

          const elemCollIndex = {
            _id: model_els_coll._userItemId,
            indexDefs: [
              {
                key: { id: 1 },
                options: {
                  name: "model_els_coll_id",
                  default_language: "english",
                },
              },
              {
                key: { source_id: 1 },
                options: {
                  name: "model_els_coll_source_id",
                  default_language: "english",
                },
              },
            ],
          };

          const typeElemCollIndex = {
            _id: model_type_el_coll._userItemId,
            indexDefs: [
              {
                key: { id: 1 },
                options: {
                  name: "typeElemsCol_id",
                  default_language: "english",
                },
              },
              {
                key: { source_id: 1 },
                options: {
                  name: "typeElemsCol_source_id",
                  default_language: "english",
                },
              },
            ],
          };

          const elemAndTypeElemIndexReses = await Promise.all([
            IafScriptEngine.createOrRecreateIndex(elemCollIndex, this.ctx),
            IafScriptEngine.createOrRecreateIndex(typeElemCollIndex, this.ctx),
          ]);
          console.log('elemAndTypeElemIndexReses:', elemAndTypeElemIndexReses)

          resolve(true);
        }
      } catch (error) {
        console.log("RefApp Error at createBIMCollectionVersion");
        console.error(error);

        reject(error);
      }
    });
  }

  async #createBIMCollections() {
    return new Promise(async (resolve, reject) => {
      try {
        const { IafScriptEngine } = this.libraries;

        console.log(
          `RefApp Creating Model Collections, orch_run_id: ${this.params.orchRunId}`
        );
        const [packagename, packagenameShort, bimpkFileId, bimpkFileVersionId] =
          await Promise.all([
            IafScriptEngine.getVar("package_name"),
            IafScriptEngine.getVar("package_name_short"),
            IafScriptEngine.getVar("bimpk_fileid"),
            IafScriptEngine.getVar("bimpk_fileVersionId"),
          ]);

        console.log(
          `RefApp packagename: ${packagename}, packagenameShort: ${packagenameShort} orch_run_id: ${this.params.orchRunId}`
        );
        //create Elements Collection
        const elementsCol = {
          _name: packagename + "_elements",
          _shortName: packagenameShort + "_ba_elem",
          _description: "Elements in BA model",
          _userType: "rvt_elements",
          _namespaces: this.ctx._namespaces,
        };

        //create Element Properties Collection
        const modelElemPropsCol = {
          _name: packagename + "_elem_props",
          _shortName: packagenameShort + "_elprops",
          _description: "Element Props in BA model",
          _userType: "rvt_element_props",
          _namespaces: this.ctx._namespaces,
        };

        //create Type Elements Collection
        const typeElemsCol = {
          _name: packagename + "_type_el",
          _shortName: packagenameShort + "_type_el",
          _description: "Type Elements in BA Check model",
          _userType: "rvt_type_elements",
          _namespaces: this.ctx._namespaces,
        };

        //create Geometry File Collection
        const geometryFilesCol = {
          _name: packagename + "_geom_file",
          _shortName: packagenameShort + "_geom_file",
          _description: "File Collection for Geometry Files",
          _userType: "bim_model_geomresources",
          _namespaces: this.ctx._namespaces,
        };

        //create Geometry View Collection
        const geometryViewsCol = {
          _name: packagename + "_geom_view",
          _shortName: packagenameShort + "_geom_view",
          _description: "Geometry Views in Model",
          _userType: "bim_model_geomviews",
          _namespaces: this.ctx._namespaces,
        };

        //create Model Data Cache Collection
        const dataCacheCol = {
          _name: packagename + "_data_cache",
          _shortName: packagenameShort + "_data_cache",
          _description: "Data cached about imported model",
          _userType: "data_cache",
          _namespaces: this.ctx._namespaces,
        };

        //create Model Composite Item
        const modelCompItem = {
          _name: packagename,
          _shortName: packagenameShort + "_modelver",
          _description: "BIM model version by transform",
          _userType: "bim_model_version",
          _namespaces: this.ctx._namespaces,
          _version: {
            _userAttributes: {
              bimpk: {
                fileId: bimpkFileId,
                fileVersionId: bimpkFileVersionId,
              },
            },
          },
        };
        console.log(
          `modelCompItem, orch_run_id: ${this.params.orchRunId}`,
          JSON.stringify(modelCompItem)
        );

        const model_els_coll = await IafScriptEngine.createCollection(elementsCol, this.ctx);
        const model_els_props_coll = await IafScriptEngine.createCollection(modelElemPropsCol, this.ctx);
        const model_type_el_coll = await IafScriptEngine.createCollection(typeElemsCol, this.ctx);
        const model_geom_file_coll = await IafScriptEngine.createCollection(geometryFilesCol, this.ctx);
        const model_geom_views_coll = await IafScriptEngine.createCollection(geometryViewsCol, this.ctx);
        const data_cache_coll = await IafScriptEngine.createCollection(dataCacheCol, this.ctx);
        const model = await IafScriptEngine.createNamedCompositeItem(modelCompItem, this.ctx);

        console.log('BimpkImport createdColls: ', JSON.stringify({
          model_els_coll: model_els_coll,
          model_els_props_coll: model_els_props_coll,
          model_type_el_coll: model_type_el_coll,
          model_geom_file_coll: model_geom_file_coll,
          model_geom_views_coll: model_geom_views_coll,
          data_cache_coll: data_cache_coll,
          model: model
        }));

        // set them in global variables
        await IafScriptEngine.setVar("model_els_coll", model_els_coll)
        await IafScriptEngine.setVar("model_els_props_coll", model_els_props_coll)
        await IafScriptEngine.setVar("model_type_el_coll", model_type_el_coll)
        await IafScriptEngine.setVar("model_geom_file_coll", model_geom_file_coll)
        await IafScriptEngine.setVar("model_geom_views_coll", model_geom_views_coll)
        await IafScriptEngine.setVar("data_cache_coll", data_cache_coll)
        await IafScriptEngine.setVar("bim_model", model)


        const elemCollIndex = {
          _id: model_els_coll._userItemId,
          indexDefs: [
            {
              key: { id: 1 },
              options: {
                name: "model_els_coll_id",
                default_language: "english",
              },
            },
            {
              key: { source_id: 1 },
              options: {
                name: "model_els_coll_source_id",
                default_language: "english",
              },
            },
          ],
        };
        const typeElemCollIndex = {
          _id: model_type_el_coll._userItemId,
          indexDefs: [
            {
              key: { id: 1 },
              options: {
                name: "typeElemsCol_id",
                default_language: "english",
              },
            },
            {
              key: { source_id: 1 },
              options: {
                name: "typeElemsCol_source_id",
                default_language: "english",
              },
            },
          ],
        };
        console.log(
          `RefApp Create or recreate index, orch_run_id: ${this.params.orchRunId}`
        );

        const elemCollIndexRes = await IafScriptEngine.createOrRecreateIndex(elemCollIndex, this.ctx)
        const typeElemCollIndexRes = await IafScriptEngine.createOrRecreateIndex(typeElemCollIndex, this.ctx)
        console.log('elemCollIndexRes:', JSON.stringify(elemCollIndexRes))
        console.log('typeElemCollIndexRes:', JSON.stringify(typeElemCollIndexRes))

        resolve(model_els_coll);
      } catch (error) {
        console.log(
          `RefApp Error at createBIMCollections, orch_run_id: ${this.params.orchRunId}`
        );
        console.error(error);

        reject(error);
      }
    });
  }

  async #extractBimpk(fileName, bimBatch, newOccurrence) {
    console.log(
      'extractBimpk fileName: ', JSON.stringify(fileName)
    );
    console.log(
      'extractBimpk bimBatch: ', JSON.stringify(bimBatch),
    );
    console.log(
      'extractBimpk newOccurrence: ', JSON.stringify(newOccurrence)
    );
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `RefApp BimpkImport extractBimpk Executing extract Bimpk function, orch_run_id: ${this.params.orchRunId}`
        );
        const { IafScriptEngine } = this.libraries;

        // Extract data
        const objects = [];
        let types = [];

        for (const obj of bimBatch.objects) {
          const batchObj = {
            package_id: obj.id,
            type_id: obj.type,
            relationships: obj.relationships,
            source_id: obj.sourceId,
            properties: obj.properties,
            source_filename: fileName,
          };
          console.log('BimpkImport extractBimpk batchObj: ', JSON.stringify(batchObj))
          objects.push(batchObj);
        }

        if (newOccurrence) {
          console.log('BimpkImport extractBimpk newOccurrance true, creating new prop map')
          this.#propMap = new Map();
          bimBatch.properties.forEach((prop) =>
            this.#propMap.set(prop.id, prop)
          );
          console.log('BimpkImport extractBimpk Creating new type map')
          this.#typeMap = new Map();

          console.log(
            `RefApp New occurrence file detected, orch_run_id: ${this.params.orchRunId}`
          );
          for (const { id, name, sourceId, properties } of bimBatch.types) {
            types.push({ id, name, source_id: sourceId, properties });
          }
          console.log(
            `RefApp BimpkImport extractBimpk About to process type extraction, typesLen: ${types?.length}, orch_run_id: ${this.params.orchRunId}`
          );
          console.log(
            `BimpkImport extractBimpk Types after types push: `, JSON.stringify(types)
          );

          for (let type of types) {
            console.log('BimpkImport extractBimpk Looping types, current type: ', JSON.stringify(type))
            for (let prop of type.properties) {
              console.log('BimpkImport extractBimpk Looping type props, current prop: ', JSON.stringify(prop))
              //const _myProp = bimBatch.properties.find(x => x.id == prop.id);
              const _myProp = this.#propMap.get(prop.id);
              prop.dName = _myProp.dName;
              if (_myProp.hasOwnProperty("Asset Category")) {
                prop.baType = _myProp["Asset Category"];
              }
            }

            type._id = await IafScriptEngine.newID("mongo", { format: "hex" });
            //console.log(JSON.stringify({"message":"Received new ID from mongo: type._id: " + type?._id}));

            type.properties = this.#groupBy(type.properties, "dName");

            if (
              this.#assetTypeMap &&
              type.properties.hasOwnProperty("Revit Family") &&
              type.properties.hasOwnProperty("Revit Type")
            ) {
              const _myRow = this.#assetTypeMap.find(
                (x) =>
                  x["Revit Family"] == type.properties["Revit Family"].val &&
                  x["Revit Type"] == type.properties["Revit Type"].val
              );

              if (_myRow) {
                type.dtCategory = _myRow.dtCategory;
                type.dtType = _myRow.dtType;
              } else {
                console.log(
                  `Could not find type in type map with matching Revit family and type for type: ${JSON.stringify(type)}`
                )
              }
            } else {
              console.log(
                `BimpkImport extractBimpk Type ${JSON.stringify(type)} has no Revit Family or Type`
              )
            }
            this.#typeMap.set(type.id, type);
          }
          console.log(
            `RefApp BimpkImport extractBimpk Type Extraction is complete, orch_run_id: ${this.params.orchRunId}`,
            JSON.stringify(types)
          );
        } else {
          console.log(
            `RefApp BimpkImport extractBimpk Old occurrence file detected. Using existing manage_type_els, orch_run_id: ${this.params.orchRunId}`
          );
          types = await IafScriptEngine.getVar("manage_type_els");
        }

        let _myProperties = [];
        console.log(`RefApp BimpkImport extractBimpk About to process objects, objectsLen: ${objects?.length}`);

        // do the same for properties in the object
        for (let obj of objects) {
          obj._id = await IafScriptEngine.newID("mongo", { format: "hex" });

          //const _myVal = types.find(x => x.id == obj.type_id);

          const _myVal = this.#typeMap.get(obj.type_id);
          obj.dtCategory = _myVal.dtCategory;
          obj.dtType = _myVal.dtType;

          if (_myVal.hasOwnProperty("baType")) {
            obj.baType = _myVal.baType;
          }

          if (obj.properties?.length > 0) {
            obj.properties.forEach((prop) => {
              //let _myProp = bimBatch.properties.find(x => x.id == prop.id);
              const _myProp = this.#propMap.get(prop.id);
              prop.dName = _myProp.dName;
            });

            obj.properties = this.#groupBy(obj.properties, "dName");
          } else {
            obj.properties = {};
          }
          const myProperty = {
            _id: obj._id,
            properties: obj.properties
          }
          _myProperties.push(myProperty);
          delete obj.properties;
        }

        console.log(
          `RefApp Property Extraction is complete, orch_run_id: ${this.params.orchRunId}`
        );
        console.log(`BimpkImport extractBimpk ${_myProperties.length} _myProperties: `, JSON.stringify(_myProperties))
        console.log(`BimpkImport extractBimpk ${objects.length} manage_els: `, JSON.stringify(objects))
        const setVars = [
          IafScriptEngine.setVar("properties", _myProperties),
          IafScriptEngine.setVar("manage_els", objects),
        ];

        if (newOccurrence) {
          setVars.push(IafScriptEngine.setVar("manage_type_els", types));
          console.log(`BimpkImport extractBimpk ${objects.length}  types: `, JSON.stringify(types))
        }
        await Promise.all(setVars);
        await IafScriptEngine.setVar(
          "manage_el_to_type_relations",
          this.#mapItemsAsRelated(objects, types, "type_id", "id")
        );

        resolve(true);
      } catch (err) {
        console.log(
          `RefApp Error at extractBimpk, orch_run_id: ${this.params.orchRunId}`
        );
        console.error(err);

        reject(err);
      }
    });
  }

  async #createRelatedItemsAndRelationships(newOccurrence, _colls) {
    console.log('BimpkImport createRelatedItemsAndRelationships running...')
    return new Promise(async (resolve, reject) => {
      try {
        const { IafScriptEngine } = this.libraries;

        console.log(
          `RefApp BimpkImport Creating Model Relations and Related Items, orch_run_id: ${this.params.orchRunId}`
        );
        const getVarCalls = [
          await IafScriptEngine.getVar("manage_els"),
          await IafScriptEngine.getVar("properties"),
          await IafScriptEngine.getVar("manage_el_to_type_relations"),
        ];
        if (newOccurrence) {
          getVarCalls.push(await IafScriptEngine.getVar("manage_type_els"));
        }
        const [manage_els, properties, relations, manage_type_els] =
          await Promise.all(getVarCalls);

        const callList = [
          IafScriptEngine.createItemsBulk(
            {
              _userItemId: _colls.model_els_coll._userItemId,
              _namespaces: this.ctx._namespaces,
              items: manage_els,
            },
            this.ctx
          ),
        ];
        console.log(
          `RefApp Create Related Collection manage_els, orch_run_id: ${this.params.orchRunId}`
        );

        if (newOccurrence) {
          callList.push(
            IafScriptEngine.createItemsBulk(
              {
                _userItemId: _colls.model_type_el_coll._userItemId,
                _namespaces: this.ctx._namespaces,
                items: manage_type_els,
              },
              this.ctx
            )
          );
          console.log(
            `RefApp Create Related Collection manage_type_els, orch_run_id: ${this.params.orchRunId}`
          );
        }

        const results = await Promise.all(callList);
        results.forEach((result, index) => {
          console.log(`BimpkImport createItemsBulk Result ${index + 1}:`, JSON.stringify(result));
        });

        console.log(
          `RefApp BimpkImport Create Related Collection properties, orch_run_id: ${this.params.orchRunId}`
        );
        console.log(
          `RefApp Create Related Collection Relations, relations: ${relations?.length} orch_run_id: ${this.params.orchRunId}`
        );

        const createItemsAsRelatedBulkAndRelations = await Promise.all([
          IafScriptEngine.createItemsAsRelatedBulk(
            {
              parentUserItemId: _colls.model_els_coll._userItemId,
              _userItemId: _colls.model_els_props_coll._userItemId,
              _namespaces: this.ctx._namespaces,
              items: properties,
            },
            this.ctx
          ),
          IafScriptEngine.createRelations(
            {
              parentUserItemId: _colls.model_els_coll._userItemId,
              _userItemId: _colls.model_type_el_coll._userItemId,
              _namespaces: this.ctx._namespaces,
              relations,
            },
            this.ctx
          ),
        ]);
        console.log(`BimpkImport createItemsAsRelatedBulkAndRelations Result: `, createItemsAsRelatedBulkAndRelations);
        console.log(`BimpkImport createItemsAsRelatedBulkAndRelations Result: `, JSON.stringify(createItemsAsRelatedBulkAndRelations));
        createItemsAsRelatedBulkAndRelations.forEach((result, index) => {
          console.log(`BimpkImport createItemsAsRelatedBulkAndRelations Result ${index + 1}:`, JSON.stringify(result));
        });

        resolve(true);
      } catch (error) {
        console.log(
          `RefApp Error at createRelatedItemsAndRelationships, orch_run_id: ${this.params.orchRunId}`
        );
        console.error(error);

        reject(error);
      }
    });
  }

  #groupBy(objectArray, property) {
    return objectArray.reduce((acc, obj) => {
      let key = obj[property];

      key = key.replace(/[\.]+/g, "");
      if (!acc[key]) {
        acc[key] = {};
      }
      // Add object to list for given key's value
      acc[key] = obj;

      return acc;
    }, {});
  }

  #mapItemsAsRelated(parentItems, relatedItems, fromField, relatedField) {
    console.log('BimpkImport mapItemsAsRelated params: ', {
      parentItems: JSON.stringify(parentItems),
      relatedItems: JSON.stringify(relatedItems),
      fromField: JSON.stringify(fromField),
      relatedField: JSON.stringify(relatedField)
    });

    // Add type checking and logging
    console.log('Type of relatedItems:', typeof relatedItems);
    console.log('Is relatedItems an array?', Array.isArray(relatedItems));

    // Convert relatedItems to an array if it's not already one
    const safeRelatedItems = Array.isArray(relatedItems)
      ? relatedItems
      : Object.values(relatedItems || {});

    console.log(
      `BimpkImport ${parentItems.length} parentItems`,
      `BimpkImport ${safeRelatedItems.length} relatedItems`,
    );

    let res = [];

    for (let i = 0, l = parentItems.length; i < l; i++) {
      let relatedRecs = [];
      let parentItem = parentItems[i];
      let fromValues = [];

      if (!parentItem[fromField] && fromField.indexOf(".") > 1) {
        fromValues = fromField
          .split(".")
          .reduce((o, i) => o[i] || [], parentItem);
      } else {
        fromValues = Array.isArray(parentItem[fromField])
          ? parentItem[fromField]
          : [parentItem[fromField]];
      }

      if (fromValues && fromValues.length > 0)
        relatedRecs = safeRelatedItems.filter((r) =>
          fromValues.includes(r[relatedField])
        );

      if (relatedRecs.length > 0) {
        res.push({
          parentItem: parentItems[i],
          relatedItems: relatedRecs,
        });
      }
    }
    console.log("RefApp mapItemsAsRelated length ", res?.length);
    console.log("RefApp BimpkImport mapItemsAsRelated res: ", JSON.stringify(res));
    return res;
  }
}

class SgpkImport {
  #assetTypeMap = null;

  constructor(params, libraries, ctx) {
    this.params = params;
    this.libraries = libraries;
    this.ctx = ctx;
  }

async initialize() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(`SGPK Sgpk import begins, orch_run_id: ${this.params.orchRunId}`);
        const { IafScriptEngine, ModelFileReader, PlatformApi, fs } = this.libraries;

        const param = {
          ...this.params.actualParams,
          filename: this.params.filename,
          orchRunId: this.params.orchRunId,
          ext: this.params.ext
        };
        console.log('param:', param);

        //creating model collections
        //set global variables first
        await IafScriptEngine.setVar("namespaces", this.ctx._namespaces);
        await IafScriptEngine.setVar("package_name", param.filename);
        await IafScriptEngine.setVar(
          "package_name_short",
          param.filename.substring(0, 11)
        )          
        await IafScriptEngine.setVar("bimpk_fileid", param._fileId);
        await IafScriptEngine.setVar("bimpk_fileVersionId", param._fileVersionId);
        
        //check for existing models
        let models;
        try {
          let res = await PlatformApi.IafItemSvc.getNamedUserItems(
            {
              query: {
                _userType: "bim_model_version",
                "_versions._userAttributes.sgpk.fileId": param._fileId,
                _itemClass: "NamedCompositeItem",
              },
            },
            this.ctx,
            {}
          );
          models = res._list;
          console.log('models: ', models)          
        } catch (e) {
          console.log('get models err: ', e)
        }

        let bim_model;
        if (models.length) {
          bim_model = models[0];
          console.log("SGPK SgpkImport init bim_model: ", bim_model)
          await IafScriptEngine.setVar("bim_model", bim_model);
        } else {
          console.log("SGPK bim model not found: ", bim_model)
        }

        if (bim_model) {
          console.time(`${this.params.orchRunId}: SGPK RefApp createBIMCollectionVersion`);
          await this.#createBIMCollectionVersion();
          console.timeEnd(
            `${this.params.orchRunId}: SGPK RefApp createBIMCollectionVersion`
          );
        } else {
          console.time(`${this.params.orchRunId}: SGPK RefApp createBIMCollections`);
          await this.#createBIMCollections();
          console.timeEnd(`${this.params.orchRunId}: SGPK RefApp createBIMCollections`);
        }

        const [
          bim_model_nci,
          model_els_coll,
          model_els_props_coll,
          model_type_el_coll,
          data_cache_coll,
          model_geom_file_coll,
          model_geom_views_coll,
        ] = await Promise.all([
          IafScriptEngine.getVar("bim_model"),
          IafScriptEngine.getVar("model_els_coll"),
          IafScriptEngine.getVar("model_els_props_coll"),
          IafScriptEngine.getVar("model_type_el_coll"),
          IafScriptEngine.getVar("data_cache_coll"),
          IafScriptEngine.getVar("model_geom_file_coll"),
          IafScriptEngine.getVar("model_geom_views_coll"),
        ]);
        let result = {
          bim_model_nci: bim_model_nci,
          model_els_coll: model_els_coll,
          model_els_props_coll: model_els_props_coll,
          model_type_el_coll: model_type_el_coll,
          data_cache_coll: data_cache_coll,
          model_geom_file_coll: model_geom_file_coll,
          model_geom_views_coll: model_geom_views_coll,
        }
        console.log("Sgpk Import Model vars result: ", result);

        const bimModelId = bim_model_nci._id;
        const relatedColls = await IafScriptEngine.addRelatedCollections(
          {
            namedCompositeItemId: bimModelId,
            relatedCollections: [
              model_els_coll._id,
              model_els_props_coll._id,
              model_type_el_coll._id,
              data_cache_coll._id,
              model_geom_file_coll._id,
              model_geom_views_coll._id,
            ],
          },
          ctx
        );
        console.log('SgpkImport relatedColls: ', relatedColls)

        const myCols = {
          model_els_coll,
          model_els_props_coll,
          model_type_el_coll,
          data_cache_coll,
          model_geom_file_coll,
          model_geom_views_coll,
        };

        try {
          this.#assetTypeMap = await IafScriptEngine.getItems(
            {
              collectionDesc: {
                _userType: "iaf_dt_type_map_defs_coll",
                _namespaces: this.ctx._namespaces,
              },
              options: {
                page: { getAllItems: true },
              },
            },
            this.ctx
          );
          console.log('SgpkImport this.#assetTypeMap: ', this.#assetTypeMap);
          console.log(
            `RefApp Asset map type received, orch_run_id: ${this.params.orchRunId} `,
            typeof this.#assetTypeMap
          );
        } catch (err) {
          console.log(
            `RefApp Type Map collection does not exist, orch_run_id: ${this.params.orchRunId}`
          );
        }

        //unzipping model file
        let zipModelFile = await ModelFileReader.downloadAndUnzipModelFile(param, ctx);
        console.log('zipModelFile retrieved:', zipModelFile);
        const { bimFilePath, manifest } = zipModelFile;
        let { files, occurrences } = manifest;
        this.processModels(bimFilePath, files, occurrences, IafScriptEngine, ModelFileReader, myCols, fs)
          .then(() => {
            return IafScriptEngine.clearVars()
            .then(() => resolve({
              filecolid: myCols.model_geom_file_coll._userItemId,
              viewcolid: myCols.model_geom_views_coll._userItemId,
              compositeitemid: bimModelId,
              bim_model_nci: bim_model_nci,
              myCollections: myCols,
              filename: param.filename,
              _fileId: param._fileId,
              _fileVersionId: param._fileVersionId,
              bimFilePath,
              manifest,
            }));
          })
          .catch(e => reject(e));
      } catch (e) {
        console.error('Initialization error:', e);
        reject(e);
      }
    });
  }

  async processModels(bimFilePath, files, occurrences, IafScriptEngine, ModelFileReader, myCols, fs) {
    const promises = [];
    for (const model of files) {
      let occs = occurrences.filter((d) => d.source === model.id);
      if (!occs?.length) {
        console.warn(`RefApp Warning: Occurrences missing for model: ${model.id}, orch_run_id: ${this.params.orchRunId}`);
        occs = [];
      }
      for (const occ of occs) {
        const filePath = bimFilePath + "/" + occ.data.scenegraph;
        try {
          const { bimBatch, endOfFile } =
            await ModelFileReader.getSgpkBatchlet(
              filePath,
              this.params.orchRunId
            );
            
          console.log('bimBatch', bimBatch)
          promises.push(
            this.#extractSgpk(bimBatch, model.id, IafScriptEngine)
              .then(modelResult => {
                console.log('model batchlet result:', modelResult);
                return this.#createRelatedItemsAndRelationships(myCols, model.id); 
              })
          );
          if (endOfFile) break;
        } catch (e) {
          console.error('Error in processModels:', e);
        }
      }
    }
    await Promise.all(promises);
  }

  async #extractSgpk(bimBatch, modelId, IafScriptEngine) {
    console.log('extractSgpk running')
    try {
      console.log(
        `RefApp SgpkImport extractSgpk Executing extract function, orch_run_id: ${this.params.orchRunId}`
      );
      // Extract data
      // extract layer types
      const bimObj = bimBatch[0];
      const rootItem = bimObj.RootItem;
      console.log('rootItem: ', rootItem)
      const sourceFileName = bimObj.SourceFileName;
      const layers = rootItem.Items;
      let layerTypes = await this.#extractLayersTypes(layers, IafScriptEngine);
      console.log('layerTypes response: ', layerTypes)

      // extract layer props and elems 
      const { layersProps, layersElems } = await this.#extractLayersPropsAndElems(layers, layerTypes, sourceFileName, IafScriptEngine);
      console.log({
        layersProps: layersProps,
        layersElems: layersElems
      });
      // extract other elems and props
      const { typeObjects, props, elems } = await this.#extractTypesPropsAndElems(layers, sourceFileName, IafScriptEngine);
      console.log({
        typeObjects: typeObjects,
        props: props,
        elems: elems
      });

      const allTypes = [ ...layerTypes, ...typeObjects ];
      const allProps = [ ...layersProps, ...props ];
      const allElems = [ ...layersElems, ...elems ];

      const allRes = {
        allTypes: allTypes,
        allProps: allProps,
        allElems: allElems,
      };
      console.log('allRes: ', allRes)
      console.log(
        `RefApp Property Extraction is complete, orch_run_id: ${this.params.orchRunId}`
      );
      console.log('SgpkImport extractSgpk', {
        _myProperties: allProps.length,
        manage_els: allElems.length,
        types: allTypes.length,
      });
      const setVars = [
        IafScriptEngine.setVar(`properties_${modelId}`, allProps),
        IafScriptEngine.setVar(`manage_els_${modelId}`, allElems),
        IafScriptEngine.setVar(`manage_type_els_${modelId}`, allTypes)
      ];
      console.log(`SgpkImport extractSgpk types len ${typeObjects.length}  types: `, allTypes)
      
      await Promise.all(setVars);
      const relations = await this.#mapItemsAsRelated(allElems, allTypes, "type_id", "id")
      console.log('mapItemsAsRelated relations: ', relations)
      await IafScriptEngine.setVar(
        `manage_el_to_type_relations_${modelId}`,
        relations
      );

      return allRes;
    } catch (err) {
      console.log(
        `RefApp Error at extractBimpk, orch_run_id: ${this.params.orchRunId}`
      );
      console.error(err);
    }
  }

  async #extractLayersTypes(layers, IafScriptEngine) {
    console.log('running extractLayersTypes layers: ', layers)
    try {
      let layersTypes = [];
      for (let layer of layers) {
        //extract type
        if (layersTypes.length === 0 || !layersTypes.some(obj => obj?.name === layer.ClassDisplayName)) {
          console.log('new layer: ', layer);
          const classDisplayName = layer.ClassDisplayName;
          const split = classDisplayName.split(':');
          const catSplit = split[0];
          const familySplit = split[1];
          const typeSplit = split[2];

          let layerType = {
            name: layer.ClassDisplayName.replace(/\s+/g, ''),
            id: layer.Id, 
            _id: await IafScriptEngine.newID("mongo", { format: "hex" }),
            source_id: null,
            properties: {
              'Revit Type': {
                val: typeSplit,
                name: 'REVIT_TYPE',
                dname: 'Revit Type',
              },
              'Revit Class': {
                val: layer.ClassName,
                name: 'REVIT_CLASS',
                dname: 'Revit Class',
              },
              'Revit Family': {
                val: familySplit,
                name: 'REVIT_FAMILY',
                dname: 'Revit Family',
              },
              'Revit Category': {
                val: catSplit,
                name: 'REVIT_CATEGORY',
                dname: 'Revit Category',
              },
            }
          }; 
          layersTypes.push(layerType)      
        }
      }
      console.log('returning layer types: ', layersTypes);
      return layersTypes;
    } catch (e) {
      console.log('extractLayersTypes err', e)
      throw e
    }    
  }

  #formatPropName(name) {
    //checks if name is already human-readable
    if (/^[A-Z][a-z]+(?: [A-Z][a-z]+)*$/.test(name)) {
        return name; 
    }
    // Replaces dots with spaces
    let formatted = name.replace(/\./g, ' ');
    // Adds spaces before uppercase letters
    formatted = formatted.replace(/([a-z])([A-Z])/g, '$1 $2');
    // Trims spaces
    const propName = formatted.trim().replace(/\s+/g, ' ');
    return propName;
  }

  async #transformPropsArrayToObject(array) {
    let transformedObject;
    if (!Array.isArray(array)) { 
      throw new Error('Expected array, but got:', typeof array);
    }
    try {
      transformedObject = array.reduce((acc, item) => {
        const rawName = item.Name;
        const formattedName = this.#formatPropName(rawName);
        if (item && item.Value !== undefined) { 
          acc[formattedName] = {
            val: item.Value,
            name: rawName,
            dName: formattedName,
            id: item.Id
          };
        }
        return acc;
      }, {}); 

    } catch (e) {
      console.log('transformed object err:', e);
    }
    console.log('Final transformed object:', transformedObject);
    return transformedObject;
  }

  async #extractLayersPropsAndElems(layers, layerTypes, sourceFileName, IafScriptEngine) {
    let layersProps = [];
    let layersElems = [];

    for (let layer of layers) {
      // extract elem
      const relatedType = layerTypes.find(t => t.name == layer.ClassDisplayName.replace(/\s+/g, ''));
      const layerElem = {
        type_id: relatedType.id,
        source_filename: sourceFileName,
        _id: await IafScriptEngine.newID("mongo", { format: "hex" }),
        package_id: layer.Id,
        source_id: null
      };
      layersElems.push(layerElem);
      const layerProp = {
        _id: layerElem._id,
        properties: await this.#transformPropsArrayToObject(layer.Properties)
      };
      layersProps.push(layerProp)
    }
    return {
      layersProps: layersProps,
      layersElems: layersElems
    }
  }

  async #extractRoomTypePropsAndElems(category, sourceFileName, IafScriptEngine){
    let elems = [];
    let props = [];
    const name = category.DisplayName;
    const typeObject = {
      name: name,
      _id: await IafScriptEngine.newID("mongo", { format: "hex" }),
      id: category.Id,
      source_id: null,
      properties: {
        'Revit Category': {
          val: category.DisplayName,
          name: 'REVIT_CATEGORY',
          dname: 'Revit Category',
        },
      }
    };
    const instances = category.Items; 
    for (let m = 0; m < instances.length; m++) {
      const elemProps = instances[m].Properties;
      const guidProps = elemProps.filter(p => p.Name == 'GUID');
      const idRegex = /^[a-zA-Z0-9]+(-[a-zA-Z0-9]+)*$/;
      const guidProp = guidProps.filter(p => typeof p.Value === 'string' && idRegex.test(p.Value));
      const elem = {
        type_id: typeObject.id,
        source_filename: sourceFileName, 
        _id: await IafScriptEngine.newID("mongo", { format: "hex" }),
        package_id: instances[m].Id,
        source_id: guidProp.Value,
      };
      elems.push(elem);
      const prop = {
        _id: elem._id,
        properties: await this.#transformPropsArrayToObject(elemProps)
      };
      props.push(prop);
    }
    const res = {
      resTypeObject: typeObject,
      resProps: props,
      resElems: elems
    };
    return res
  }

  async #extractTypesPropsAndElems(layers, sourceFileName, IafScriptEngine){
    let typeObjects = [];
    let props = [];
    let elems = [];
    const ignoredNames = ['<Room Separation>', 'Center line', 'Center Line', 'Lines'];
    for (let i = 0; i < layers.length; i++) {
      try {
        if (layers[i].Items) {
          const categories = layers[i].Items;
          for (let j = 0; j < categories.length; j++) {
            try {
              if (!ignoredNames.includes(categories[j].ClassDisplayName)) {
                if (categories[j].DisplayName == 'Rooms') {
                  const {resTypeObject, resProps, resElems} = await this.#extractRoomTypePropsAndElems(categories[j], sourceFileName, IafScriptEngine);
                  typeObjects.push(resTypeObject);
                  elems.push(...resElems);
                  props.push(...resProps);
                } else {
                  if (categories[j].Items) {
                    const families = categories[j].Items;
                    for (let k = 0; k < families.length; k++) {
                      if (families[k].Items) {
                        const types = families[k].Items;
                        console.log('types', types);
                        for (let l = 0; l < types.length; l++) {
                          const name = [categories[j].DisplayName, families[k].DisplayName, types[l].DisplayName].join(':');
                          const typeExists = typeObjects.some(obj => obj.name === name);
                          if (!typeExists) {
                            const typeObject = {
                              name: name.replace(/\s+/g, ''),
                              _id: await IafScriptEngine.newID("mongo", { format: "hex" }),
                              id: types[l].Id,
                              source_id: null,
                              properties: {
                                'Revit Type': {
                                  val: types[l].DisplayName,
                                  name: 'REVIT_TYPE',
                                  dname: 'Revit Type',
                                },
                                // Revit Class: {
                                //     val: classes[m].ClassName,
                                //     name: 'REVIT_CLASS',
                                //     dname: 'Revit Class',
                                // },
                                'Revit Family': {
                                  val: families[k].DisplayName,
                                  name: 'REVIT_FAMILY',
                                  dname: 'Revit Family',
                                },
                                'Revit Category': {
                                  val: categories[j].DisplayName,
                                  name: 'REVIT_CATEGORY',
                                  dname: 'Revit Category',
                                },
                              }
                            }  
                            typeObjects.push(typeObject);                      
                          }
                          if (types[l].Items) {
                            const instances = types[l].Items;
                            for (let m = 0; m < instances.length; m++) {
                              const relatedType = typeObjects.find(t => t.name == instances[m].ClassDisplayName.replace(/\s+/g, ''));
                              const elemProps = instances[m].Properties;
                              const guidProps = elemProps.filter(p => p.Name == 'GUID');
                              console.log('')
                              const idRegex = /^[a-zA-Z0-9]+(-[a-zA-Z0-9]+)*$/;
                              const guidProp = guidProps.filter(p => typeof p.Value === 'string' && idRegex.test(p.Value));
                              const elem = {
                                type_id: relatedType.id,
                                source_filename: sourceFileName, 
                                _id: await IafScriptEngine.newID("mongo", { format: "hex" }),
                                package_id: instances[m].Id,
                                source_id: guidProp.Value,
                              };
                              elems.push(elem);

                              const prop = {
                                _id: elem._id,
                                properties: await this.#transformPropsArrayToObject(elemProps)
                              };
                              console.log('extractTypesPropsAndElems pushing prop: ', prop)
                              props.push(prop);
                            }
                          } else {
                            console.log('extractTypesPropsAndElems no instances for type: ', types[l].Id)
                          }
                        }
                      } else {
                        console.log('extractTypesPropsAndElems no types for family: ', families[k].Id)
                      }
                    }
                  } else {
                    console.log('extractTypesPropsAndElems No families for category: ', categories[j].Id)
                  } 
                }              
              }
            } catch (e) {
              console.log('extractTypesPropsAndElems categories try error: ', e)
            }
          }            
        } else {
          console.log('extractTypesPropsAndElems categories missing from layer: ', layers[i].Id)
        }
      
      } catch (e) {
        console.log('extractTypesPropsAndElems layers try err: ', e)
      }
    }
    return {
      typeObjects: typeObjects,
      props: props,
      elems: elems
    }
  }

  async #mapItemsAsRelated(parentItems, relatedItems, fromField, relatedField) {
    console.log('SgpkImport mapItemsAsRelated running');
    console.log('SgpkImport mapItemsAsRelated params: ', {
      parentItems: parentItems,
      relatedItems: relatedItems,
      fromField: fromField,
      relatedField: relatedField
    });

    // Add type checking and logging
    console.log('Type of relatedItems:', typeof relatedItems);
    console.log('Is relatedItems an array?', Array.isArray(relatedItems));

    // Convert relatedItems to an array if it's not already one
    const safeRelatedItems = Array.isArray(relatedItems)
      ? relatedItems
      : Object.values(relatedItems || {});

    console.log(
      `SgpkImport ${parentItems.length} parentItems`,
      `SgpkImport ${safeRelatedItems.length} relatedItems`,
    );

    let res = [];

    for (let i = 0, l = parentItems.length; i < l; i++) {
      let relatedRecs = [];
      let parentItem = parentItems[i];
      let fromValues = [];

      if (!parentItem[fromField] && fromField.indexOf(".") > 1) {
        fromValues = fromField
          .split(".")
          .reduce((o, i) => o[i] || [], parentItem);
      } else {
        fromValues = Array.isArray(parentItem[fromField])
          ? parentItem[fromField]
          : [parentItem[fromField]];
      }

      if (fromValues && fromValues.length > 0)
        relatedRecs = safeRelatedItems.filter((r) =>
          fromValues.includes(r[relatedField])
        );

      if (relatedRecs.length > 0) {
        res.push({
          parentItem: parentItems[i],
          relatedItems: relatedRecs,
        });
      }
    }
    console.log("RefApp mapItemsAsRelated length ", res?.length);
    console.log("RefApp SgpkImport mapItemsAsRelated res: ", JSON.stringify(res));
    return res;
  }

  async #createRelatedItemsAndRelationships(_colls, modelId) {
    console.log('SgpkImport createRelatedItemsAndRelationships running...')
    return new Promise(async (resolve, reject) => {
      try {
        const { IafScriptEngine } = this.libraries;

        console.log(
          `RefApp SgpkImport Creating Model Relations and Related Items, orch_run_id: ${this.params.orchRunId}`
        );
        const getVarCalls = [
          await IafScriptEngine.getVar(`manage_els_${modelId}`),
          await IafScriptEngine.getVar(`properties_${modelId}`),
          await IafScriptEngine.getVar(`manage_el_to_type_relations_${modelId}`),
          await IafScriptEngine.getVar(`manage_type_els_${modelId}`)
        ];
     
        const [manage_els, properties, relations, manage_type_els] =
          await Promise.all(getVarCalls);

        const callList = [
          IafScriptEngine.createItemsBulk(
            {
              _userItemId: _colls.model_els_coll._userItemId,
              _namespaces: this.ctx._namespaces,
              items: manage_els,
            },
            this.ctx
          ),
        ];
        console.log(
          `RefApp Create Related Collection manage_els, orch_run_id: ${this.params.orchRunId}`
        );

        callList.push(
          IafScriptEngine.createItemsBulk(
            {
              _userItemId: _colls.model_type_el_coll._userItemId,
              _namespaces: this.ctx._namespaces,
              items: manage_type_els,
            },
            this.ctx
          )
        );
        console.log(
          `RefApp Create Related Collection manage_type_els, orch_run_id: ${this.params.orchRunId}`
        );
        

        const results = await Promise.all(callList);
        results.forEach((result, index) => {
          console.log(`SgpkImport createItemsBulk Result ${index + 1}:`, JSON.stringify(result));
        });

        console.log(
          `RefApp SgpkImport Create Related Collection properties, orch_run_id: ${this.params.orchRunId}`
        );
        console.log(
          `RefApp Create Related Collection Relations, relations: ${relations?.length} orch_run_id: ${this.params.orchRunId}`
        );

        const createItemsAsRelatedBulkAndRelations = await Promise.all([
          IafScriptEngine.createItemsAsRelatedBulk(
            {
              parentUserItemId: _colls.model_els_coll._userItemId,
              _userItemId: _colls.model_els_props_coll._userItemId,
              _namespaces: this.ctx._namespaces,
              items: properties,
            },
            this.ctx
          ),
          IafScriptEngine.createRelations(
            {
              parentUserItemId: _colls.model_els_coll._userItemId,
              _userItemId: _colls.model_type_el_coll._userItemId,
              _namespaces: this.ctx._namespaces,
              relations,
            },
            this.ctx
          ),
        ]);
        console.log(`SgpkImport createItemsAsRelatedBulkAndRelations Result: `, createItemsAsRelatedBulkAndRelations);
        console.log(`SgpkImport createItemsAsRelatedBulkAndRelations Result: `, JSON.stringify(createItemsAsRelatedBulkAndRelations));
        createItemsAsRelatedBulkAndRelations.forEach((result, index) => {
          console.log(`SgpkImport createItemsAsRelatedBulkAndRelations Result ${index + 1}:`, JSON.stringify(result));
        });

        resolve(true);
      } catch (error) {
        console.log(
          `RefApp Error at createRelatedItemsAndRelationships, orch_run_id: ${this.params.orchRunId}`
        );
        console.error(error);

        reject(error);
      }
    });
  }

  async #createBIMCollections() {
    return new Promise(async (resolve, reject) => {
      try {
        const { IafScriptEngine } = this.libraries;

        console.log(
          `SGPK RefApp Creating Model Collections, orch_run_id: ${this.params.orchRunId}`
        );
        const [packagename, packagenameShort, bimpkFileId, bimpkFileVersionId] =
          await Promise.all([
            IafScriptEngine.getVar("package_name"),
            IafScriptEngine.getVar("package_name_short"),
            IafScriptEngine.getVar("bimpk_fileid"),
            IafScriptEngine.getVar("bimpk_fileVersionId"),
          ]);

        //create Elements Collection
        const elementsCol = {
          _name: packagename + "_elements",
          _shortName: packagenameShort + "_ba_elem",
          _description: "Elements in BA model",
          _userType: "rvt_elements",
          _namespaces: this.ctx._namespaces,
        };

        //create Element Properties Collection
        const modelElemPropsCol = {
          _name: packagename + "_elem_props",
          _shortName: packagenameShort + "_elprops",
          _description: "Element Props in BA model",
          _userType: "rvt_element_props",
          _namespaces: this.ctx._namespaces,
        };

        //create Type Elements Collection
        const typeElemsCol = {
          _name: packagename + "_type_el",
          _shortName: packagenameShort + "_type_el",
          _description: "Type Elements in BA Check model",
          _userType: "rvt_type_elements",
          _namespaces: this.ctx._namespaces,
        };
        //create Geometry File Collection
        const geometryFilesCol = {
          _name: packagename + "_geom_file",
          _shortName: packagenameShort + "_geom_file",
          _description: "File Collection for Geometry Files",
          _userType: "bim_model_geomresources",
          _namespaces: this.ctx._namespaces,
        };
        //create Geometry View Collection
        const geometryViewsCol = {
          _name: packagename + "_geom_view",
          _shortName: packagenameShort + "_geom_view",
          _description: "Geometry Views in Model",
          _userType: "bim_model_geomviews",
          _namespaces: this.ctx._namespaces,
        };

        //create Model Data Cache Collection
        const dataCacheCol = {
          _name: packagename + "_data_cache",
          _shortName: packagenameShort + "_data_cache",
          _description: "Data cached about imported model",
          _userType: "data_cache",
          _namespaces: this.ctx._namespaces,
        };

        //create Model Composite Item
        const modelCompItem = {
          _name: packagename,
          _shortName: packagenameShort + "_modelver",
          _description: "BIM model version by transform",
          _userType: "bim_model_version",
          _namespaces: this.ctx._namespaces,
          _version: {
            _userAttributes: {
              sgpk: {
                fileId: bimpkFileId,
                fileVersionId: bimpkFileVersionId,
              },
            },
          },
        };
        console.log(
          `modelCompItem, orch_run_id: ${this.params.orchRunId}`,
          JSON.stringify(modelCompItem)
        );
        const model_els_coll = await IafScriptEngine.createCollection(elementsCol, this.ctx);
        const model_els_props_coll = await IafScriptEngine.createCollection(modelElemPropsCol, this.ctx);
        const model_type_el_coll = await IafScriptEngine.createCollection(typeElemsCol, this.ctx);
        const model_geom_file_coll = await IafScriptEngine.createCollection(geometryFilesCol, this.ctx);
        const model_geom_views_coll = await IafScriptEngine.createCollection(geometryViewsCol, this.ctx);
        const data_cache_coll = await IafScriptEngine.createCollection(dataCacheCol, this.ctx);
        const model = await IafScriptEngine.createNamedCompositeItem(modelCompItem, this.ctx);

        console.log('BimpkImport createdColls: ', JSON.stringify({
          model_els_coll: model_els_coll,
          model_els_props_coll: model_els_props_coll,
          model_type_el_coll: model_type_el_coll,
          model_geom_file_coll: model_geom_file_coll,
          model_geom_views_coll: model_geom_views_coll,
          data_cache_coll: data_cache_coll,
          model: model
        }));

        // set them in global variables
        await IafScriptEngine.setVar("model_els_coll", model_els_coll)
        await IafScriptEngine.setVar("model_els_props_coll", model_els_props_coll)
        await IafScriptEngine.setVar("model_type_el_coll", model_type_el_coll)
        await IafScriptEngine.setVar("model_geom_file_coll", model_geom_file_coll)
        await IafScriptEngine.setVar("model_geom_views_coll", model_geom_views_coll)
        await IafScriptEngine.setVar("data_cache_coll", data_cache_coll)
        await IafScriptEngine.setVar("bim_model", model)


        const elemCollIndex = {
          _id: model_els_coll._userItemId,
          indexDefs: [
            {
              key: { id: 1 },
              options: {
                name: "model_els_coll_id",
                default_language: "english",
              },
            },
            {
              key: { source_id: 1 },
              options: {
                name: "model_els_coll_source_id",
                default_language: "english",
              },
            },
          ],
        };
        const typeElemCollIndex = {
          _id: model_type_el_coll._userItemId,
          indexDefs: [
            {
              key: { id: 1 },
              options: {
                name: "typeElemsCol_id",
                default_language: "english",
              },
            },
            {
              key: { source_id: 1 },
              options: {
                name: "typeElemsCol_source_id",
                default_language: "english",
              },
            },
          ],
        };
        console.log(
          `RefApp Create or recreate index, orch_run_id: ${this.params.orchRunId}`
        );

        const elemCollIndexRes = await IafScriptEngine.createOrRecreateIndex(elemCollIndex, this.ctx)
        const typeElemCollIndexRes = await IafScriptEngine.createOrRecreateIndex(typeElemCollIndex, this.ctx)
        console.log('elemCollIndexRes:', JSON.stringify(elemCollIndexRes))
        console.log('typeElemCollIndexRes:', JSON.stringify(typeElemCollIndexRes))

        resolve(model_els_coll);
      } catch (error) {
        console.error(
          `SGPK RefApp Error at createBIMCollections, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  async #createBIMCollectionVersion() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `RefApp Found Previous Model Creating Versions, orch_run_id: ${this.params.orchRunId}`
        );
        const { IafScriptEngine } = this.libraries;
        const [
          bimModel,
          bimpkFileId,
          bimpkFileVersionId,
          packagename,
          packagenameShort,
        ] = await Promise.all([
          IafScriptEngine.getVar("bim_model"),
          IafScriptEngine.getVar("bimpk_fileid"),
          IafScriptEngine.getVar("bimpk_fileVersionId"),
          IafScriptEngine.getVar("package_name"),
          IafScriptEngine.getVar("package_name_short"),
        ]);
        const newModelVer = {
          namedUserItemId: bimModel._id,
          _userAttributes: {
            sgpk: {
              fileId: bimpkFileId,
              fileVersionId: bimpkFileVersionId,
            },
          },
        };
        const modelRelatedCollection = await IafScriptEngine.getCollectionsInComposite(
          bimModel._id,
          null,
          this.ctx
        )

        //IafScriptEngine.createNamedUserItemVersion(newModelVer, this.ctx),
        console.log(`RefApp modelRelatedCollection, orch_run_id: ${this.params.orchRunId}`, JSON.stringify(modelRelatedCollection));
        console.log(`RefApp Created BIM Collection Version bim_model version, orch_run_id: ${this.params.orchRunId}`);

        let model_els_coll,
          model_els_props_coll,
          model_type_el_coll,
          model_geom_file_coll,
          model_geom_views_coll,
          data_cache_coll;

        if (modelRelatedCollection.length) {

          for (const obj of modelRelatedCollection) {
            if (obj._userType === "rvt_elements" && !model_els_coll) {
              model_els_coll = obj;
            } else if (
              obj._userType === "rvt_element_props" &&
              !model_els_props_coll
            ) {
              model_els_props_coll = obj;
            } else if (
              obj._userType === "rvt_type_elements" &&
              !model_type_el_coll
            ) {
              model_type_el_coll = obj;
            } else if (
              obj._userType === "bim_model_geomresources" &&
              !model_geom_file_coll
            ) {
              model_geom_file_coll = obj;
            } else if (
              obj._userType === "bim_model_geomviews" &&
              !model_geom_views_coll
            ) {
              model_geom_views_coll = obj;
            } else if (obj._userType === "data_cache" && !data_cache_coll) {
              data_cache_coll = obj;
            }
          }

          if (!data_cache_coll) {
            const data_cache_coll_def = {
              _name: packagename + "_data_cache",
              _shortName: packagenameShort + "_data_cache",
              _description: "Data cached about imported model",
              _userType: "data_cache",
              _namespaces: this.ctx._namespaces,
            };

            data_cache_coll = await IafScriptEngine.createCollection(
              data_cache_coll_def,
              this.ctx
            );

            console.log(
              `RefApp Created Model Data Cache, orch_run_id: ${this.params.orchRunId}`
            );
          }

          console.log(
            `RefApp Created Model versions, orch_run_id: ${this.params.orchRunId}`
          );
          // create the versions
          await Promise.all([
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_els_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_els_props_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_type_el_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: data_cache_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_geom_file_coll._userItemId },
              this.ctx
            ),
            IafScriptEngine.createNamedUserItemVersion(
              { namedUserItemId: model_geom_views_coll._userItemId },
              this.ctx
            )
          ]);

          // set them in global variables

          await IafScriptEngine.setVar("model_els_coll", model_els_coll)
          await IafScriptEngine.setVar("model_els_props_coll", model_els_props_coll)
          await IafScriptEngine.setVar("model_type_el_coll", model_type_el_coll)
          await IafScriptEngine.setVar("data_cache_coll", data_cache_coll)
          await IafScriptEngine.setVar("model_geom_file_coll", model_geom_file_coll)
          await IafScriptEngine.setVar("model_geom_views_coll", model_geom_views_coll)

          const elemCollIndex = {
            _id: model_els_coll._userItemId,
            indexDefs: [
              {
                key: { id: 1 },
                options: {
                  name: "model_els_coll_id",
                  default_language: "english",
                },
              },
              {
                key: { source_id: 1 },
                options: {
                  name: "model_els_coll_source_id",
                  default_language: "english",
                },
              },
            ],
          };

          const typeElemCollIndex = {
            _id: model_type_el_coll._userItemId,
            indexDefs: [
              {
                key: { id: 1 },
                options: {
                  name: "typeElemsCol_id",
                  default_language: "english",
                },
              },
              {
                key: { source_id: 1 },
                options: {
                  name: "typeElemsCol_source_id",
                  default_language: "english",
                },
              },
            ],
          };

          const elemAndTypeElemIndexReses = await Promise.all([
            IafScriptEngine.createOrRecreateIndex(elemCollIndex, this.ctx),
            IafScriptEngine.createOrRecreateIndex(typeElemCollIndex, this.ctx),
          ]);
          console.log('elemAndTypeElemIndexReses:', elemAndTypeElemIndexReses)

          resolve(true);
        }
      } catch (error) {
        console.log("RefApp Error at createBIMCollectionVersion");
        console.error(error);

        reject(error);
      }
    });
  }

  async #addRelatedCollections(_colls) {
    return new Promise(async (resolve, reject) => {
      try {
        const { IafScriptEngine } = this.libraries;

        console.log(
          `SGPK RefApp Creating Model Relations and Related Items, orch_run_id: ${this.params.orchRunId}`
        );
        const { _id: bim_model_id } = await IafScriptEngine.getVar("bim_model");

        await IafScriptEngine.addRelatedCollections(
          {
            namedCompositeItemId: bim_model_id,
            relatedCollections: [
              _colls.model_geom_file_coll._userItemId,
              _colls.model_geom_views_coll._userItemId,
            ],
          },
          ctx
        );

        await IafScriptEngine.setVar("outparams", {
          filecolid: _colls.model_geom_file_coll._userItemId,
          viewcolid: _colls.model_geom_views_coll._userItemId,
          compositeitemid: bim_model_id,
        });

        resolve(true);
      } catch (error) {
        console.error(
          `SGPK RefApp Error at addRelatedCollections, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }
}

class SczImport {
  constructor(params, libraries, ctx) {
    this.params = params;
    this.libraries = libraries;
    this.ctx = ctx;
  }

  async processScz() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing processScz method, orch_run_id: ${this.params.orchRunId}`
        );

        const { PlatformApi, _ } = this.libraries;
        const { manifest } = this.params.result;
        const graphicsFilePath = await _.get(
          await _.find(
            manifest.occurrences,
            (oc) => oc.data && oc.data.graphics
          ),
          "data.graphics"
        );
        const graphics2dFilePath =
          (await _.get(
            await _.find(
              manifest.occurrences,
              (oc) => oc.data && oc.data.graphics
            ),
            "data.graphics2d"
          )) ||
          (await _.get(
            await _.find(
              manifest.occurrences,
              (oc) => oc.data && oc.data.graphics2d
            ),
            "data.graphics2d"
          ));

        let folderName;
        let folder;
        let viewItems = [];

        if (graphicsFilePath || graphics2dFilePath) {
          folderName = "geometryData_" + this.params.context.id;
          folder = await PlatformApi.IafFileSvc.addFolder(
            folderName,
            this.params.context._namespaces[0],
            undefined,
            this.params.context
          );

          await this.#updateFileDetailsInModel(folder._id);
        }

        if (graphicsFilePath) {
          const viewItems3d = await this.#process3D(graphicsFilePath, folder);

          if (viewItems3d && viewItems3d.length > 0) {
            viewItems = viewItems.concat(viewItems3d);
          }
        }

        if (graphics2dFilePath) {
          const viewItems2d = await this.#process2D(graphics2dFilePath, folder);

          if (viewItems2d && viewItems2d.length > 0) {
            viewItems = viewItems.concat(viewItems2d);
          }
        }

        await this.#updateViewableResources(viewItems);

        resolve(true);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at processScz method, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  async #updateFileDetailsInModel(folderId) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing updateFileDetailsInModel method, orch_run_id: ${this.params.orchRunId}`
        );

        const { PlatformApi, _, ModelFileReader } = this.libraries;
        const { manifest } = this.params.result;

        let thumbnailPath =
          (await _.get(
            await _.find(manifest.occurrences, (oc) => oc.data),
            "data.thumbnail"
          )) || "thumbnail.png";

        thumbnailPath = this.params.result.bimFilePath + "/" + thumbnailPath;

        const uploadParams = {
          filePath: thumbnailPath,
          _namespaces: this.params.context._namespaces[0],
          folderId,
          tags: undefined,
          ctx: this.params.context,
          opts: {},
        };

        const fileUpload = await ModelFileReader.uploadSmallFile(uploadParams);

        if (fileUpload?.isFileuploaded) {
          const { fileInfo } = fileUpload;
          const userAttributes = {
            thumbnail: {
              fileId: fileInfo[0]._id,
              fileVersionId: fileInfo[0]._tipId,
            },
            [this.params.ext]: {
              fileId: this.params.result._fileId,
              fileVersionId: this.params.result._fileVersionId,
            },
          };

          const compItem = await PlatformApi.IafItemSvc.getNamedUserItem(
            this.params.result.compositeitemid,
            this.params.context
          );
          const version = compItem._versions[0];

          if (version.hasOwnProperty("_userAttributes")) {
            Object.assign(version._userAttributes, userAttributes);
          } else {
            version._userAttributes = userAttributes;
          }

          await PlatformApi.IafItemSvc.updateNamedUserItemVersion(
            compItem._id,
            version._id,
            version,
            this.params.context
          );
        }

        resolve(true);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at updateFileDetailsInModel method, orch_run_id: ${params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  async #process3D(graphicsFilePath, folder) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing process3D method, orch_run_id: ${this.params.orchRunId}`
        );
        const { path, ModelFileReader } = this.libraries;
        let viewItem = {
          title: "3dGraphics",
          aspect: "View3d",
        };

        //Upload Scz
        const filePath =
          this.params.result.bimFilePath + "/" + graphicsFilePath;
        const filename = this.params.result.filename + ".scz";
        const opts = { filename };
        const tags = ["#invgraphicsfile#"];

        //Scz file upload
        const uploadParams = {
          filePath,
          opts,
          tags,
          folder,
          context: this.params.context,
        };
        const uploadRes = await ModelFileReader.uploadLargeFile(uploadParams);

        if (uploadRes?.isFileuploaded) {
          await this.#cacheFile(uploadRes.file?._id);
          const sczFileItem = await this.#createFileItem(uploadRes.file);

          viewItem.viewableResources = [sczFileItem._id];
        }

        //Upload mapping file
        const hoops_node_mapping_file = "hoops_node_mapping.json";
        const mappingFilePath = this.#updatePath(
          filePath,
          hoops_node_mapping_file
        );
        const mapFileUploadParams = {
          filePath: mappingFilePath,
          opts: { filename: hoops_node_mapping_file },
          tags: [],
          folder,
          context: this.params.context,
        };
        const mapFile = await ModelFileReader.uploadLargeFile(
          mapFileUploadParams
        );

        if (mapFile?.isFileuploaded) {
          const mappingFileItem = await this.#createFileItem(mapFile.file);

          viewItem.resources = [mappingFileItem._id];
        }

        resolve([viewItem]);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at process3D method, orch_run_id: ${params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  async #process2D(graphics2dFilePath, folder) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing process2D method, orch_run_id: ${this.params.orchRunId}`
        );
        const { ModelFileReader } = this.libraries;

        if (!graphics2dFilePath) return [];

        //Upload Scz
        const filePath =
          this.params.result.bimFilePath + "/" + graphics2dFilePath;
        const filename = this.params.result.filename + "_2d.scz";
        const opts = { filename };
        const tags = ["#invgraphicsfile#"];

        //Scz file upload
        const uploadParams = {
          filePath,
          opts,
          tags,
          folder,
          context: this.params.context,
        };
        const uploadRes = await ModelFileReader.uploadLargeFile(uploadParams);
        let sczFileItem = {};

        if (uploadRes?.isFileuploaded) {
          await this.#cacheFile(uploadRes.file?._id);
          sczFileItem = await this.#createFileItem(uploadRes.file);
        }

        let viewItems = [];
        //Upload mapping file
        const hoops_node_mapping_file = "hoops_node_mapping2d.json";
        const mappingFilePath = this.#updatePath(
          filePath,
          hoops_node_mapping_file
        );
        const mapFileUploadParams = {
          filePath: mappingFilePath,
          opts: { filename: hoops_node_mapping_file },
          tags: [],
          folder,
          readFile: true,
          context: this.params.context,
        };
        const mappingFile = await ModelFileReader.uploadLargeFile(
          mapFileUploadParams
        );

        if (mappingFile?.isFileuploaded) {
          const mappingFileItem = await this.#createFileItem(mappingFile.file);
          const mapping2D = JSON.parse(mappingFile.rawFile);

          Object.keys(mapping2D).forEach((key) => {
            viewItems.push({
              title: key,
              aspect: "View2d",
              viewableResources: [sczFileItem._id],
              resources: [mappingFileItem._id],
            });
          });
        }

        resolve(viewItems);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at process2D method, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  async #cacheFile(_fileId) {
    return new Promise(async (resolve, reject) => {
      const { PlatformApi } = this.libraries;

      try {
        await PlatformApi.IafGraphicsSvc.cacheGraphicsFiles(
          [{ _fileId }],
          this.params.context
        );

        console.log(
          `SGPK RefApp Successfully cached file_id: ${_fileId} in graphics service, orch_run_id: ${this.params.orchRunId}`
        );

        resolve(true);
      } catch (error) {
        reject(error);
      }
    });
  }

  #updatePath(filePath, fileName) {
    try {
      let splitPaths = filePath.split("/");

      splitPaths[splitPaths.length - 1] = fileName;
      const modifiedFilePath = splitPaths.join("/");

      return modifiedFilePath;
    } catch (error) {
      console.log(
        `SGPK RefApp Error at #updatePath, orch_run_id: ${this.params.orchRunId}`,
        error
      );

      throw error;
    }
  }

  async #createFileItem(fileObj) {
    return new Promise(async (resolve, reject) => {
      try {
        const fileAttr = {
          _fileId: fileObj._id,
          _fileVersionId: fileObj._tipId,
          filename: fileObj._name,
        };

        const { _list = [] } =
          (await PlatformApi.IafItemSvc.createRelatedItems(
            this.params.result.filecolid,
            [fileAttr],
            this.params.context
          )) || {};

        resolve(_list[0]);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at createFileItem method, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  async #updateViewableResources(viewItems) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing updateViewableResources method, orch_run_id: ${this.params.orchRunId}`
        );

        const { PlatformApi } = this.libraries;

        if (!viewItems || viewItems.length <= 0) return resolve(null);

        for (let viewItem of viewItems) {
          let resViewItem = await PlatformApi.IafItemSvc.createRelatedItems(
            this.params.result.viewcolid,
            [viewItem],
            this.params.context
          );
          resViewItem = resViewItem._list[0];

          let relationShips = [];
          let relatedToIds = viewItem.viewableResources
            ? viewItem.viewableResources
            : [];
          relatedToIds = relatedToIds
            ? relatedToIds.concat(viewItem.resources ? viewItem.resources : [])
            : [];

          if (relatedToIds.length > 0) {
            const relations = {
              _relatedFromId: resViewItem._id,
              _relatedUserItemId: this.params.result.filecolid,
              _relatedToIds: relatedToIds,
            };

            relationShips.push(relations);

            await PlatformApi.IafItemSvc.addRelationsBulk(
              this.params.result.viewcolid,
              relationShips,
              this.params.context
            );
          }
        }

        resolve(true);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at updateViewableResources method, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }
}

class Helper {
  constructor(params, libraries) {
    this.params = params;
    this.libraries = libraries;
  }

  async getFileMetaData(fileId) {
    return new Promise(async (resolve, reject) => {
      try {
        if (!fileId) throw new Error("SGPK RefApp _fileId is mandatory");
        const { PlatformApi } = this.libraries;
        const { _name } =
          (await PlatformApi.IafFileSvc.getFile(fileId, this.params.context)) ||
          {};

        if (!_name) {
          throw new Error(
            `SGPK RefApp Couldn't retrieve file information for _fileId: ${fileId}`
          );
        }
        const fileMeta = {
          filename: _name.slice(0, _name.lastIndexOf(".")),
          ext: _name.split(".")?.pop(),
        };
        resolve(fileMeta);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at getFileMetaData method, orch_run_id: ${this.params.orchRunId}`,
          error
        );
        reject(error);
      }
    });
  }
}

class SczTarget {
  constructor(params, libraries, ctx) {
    this.params = params;
    this.libraries = libraries;
    this.ctx = ctx;
  }

  async intialize() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log("SGPK RefApp scz init")
        const resources = await this.#processScz();

        await this.#updateViewableResources(resources);

        resolve(true);
      } catch (err) {
        reject(err);
      }
    });
  }

  async #processScz() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing processScz method in SczTarget, orch_run_id: ${this.params.orchRunId}`
        );
        let viewItems = [];
        viewItems = await this.#processGraphics();

        let viewItems2d = [];
        viewItems2d = await this.#processGraphics2d();

        viewItems = viewItems.concat(viewItems2d);

        resolve(viewItems);
      } catch (error) {
        reject(error);
      }
    });
  }

  async #processGraphics() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing processGraphics method in SczTarget, orch_run_id: ${this.params.orchRunId}`
        );

        const { manifest } = this.params.result;
        const { files: manifestFiles, occurrences } = manifest;
        let folderName;
        let viewItems = [];

        for (const maniFile of manifestFiles) {
          const layers = maniFile.layers;
          // find occurances
          const occs = occurrences.filter((d) => d.source === maniFile.id);
          const graphicsFileTitle = this.#removeFileExtension(maniFile.name);

          for (const occ of occs) {
            const graphicsFilePath = occ.data.graphics;
            const graphics2dFilePath = occ.data.graphics2d;
            const modelBounding = occ.modelBounding;
            // folderName = 'geometryData_' + params.context.id + "_" + occ.id;

            const viewItemsOcc = await this.#addViewItems(
              graphicsFilePath,
              graphics2dFilePath,
              folderName,
              graphicsFileTitle,
              layers,
              modelBounding
            );

            viewItems = viewItems.concat(viewItemsOcc);
          }
          //maniFile.occurences = occs
        }

        resolve(viewItems);
      } catch (error) {
        reject(error);
      }
    });
  }

  async #addViewItems(
    graphicsFilePath,
    graphics2dFilePath,
    folderName,
    graphicsFileTitle,
    layers,
    modelBounding
  ) {
    return new Promise(async (resolve, reject) => {
      try {
        const { PlatformApi } = this.libraries;
        let folder;
        let viewItems = [];

        if (graphicsFilePath || graphics2dFilePath) {
          folderName = "geometryData_" + this.params.context.id;
          folder = await PlatformApi.IafFileSvc.addFolder(
            folderName,
            this.params.context._namespaces[0],
            undefined,
            this.params.context
          );

          await this.#updateFileDetailsInModel(folder._id);
        }

        if (graphicsFilePath) {
          const viewItems3d = await this.#process3D(
            graphicsFilePath,
            folder,
            graphicsFileTitle,
            layers,
            modelBounding
          );

          if (viewItems3d && viewItems3d.length > 0) {
            viewItems = viewItems.concat(viewItems3d);
          }
        }

        if (graphics2dFilePath) {
          const viewItems2d = await this.#process2D(
            graphics2dFilePath,
            folder,
            graphicsFileTitle
          );

          if (viewItems2d && viewItems2d.length > 0) {
            viewItems = viewItems.concat(viewItems2d);
          }
        }

        resolve(viewItems);
      } catch (error) {
        console.error(error);
        reject(error);
      }
    });
  }

  async #processGraphics2d() {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing processGraphics2d method, orch_run_id: ${this.params.orchRunId}`
        );
        const { view2ds } = this.params.result.manifest;

        if (!view2ds) {
          console.warn(
            `SGPK RefApp In processGraphics2d, view2ds is found empty, skipping, orch_run_id: ${this.params.orchRunId}`
          );
          return resolve([]);
        }
        let folderName;
        let viewItems = [];

        for (const {
          name: graphicsFileTitle,
          graphics2d: graphics2dFilePath,
        } of view2ds) {
          const viewItemsOcc = await this.#addViewItems2d(
            graphics2dFilePath,
            folderName,
            graphicsFileTitle
          );

          viewItems = viewItems.concat(viewItemsOcc);
        }

        resolve(viewItems);
      } catch (error) {
        reject(error);
      }
    });
  }

  async #addViewItems2d(graphics2dFilePath, folderName, graphicsFileTitle) {
    return new Promise(async (resolve, reject) => {
      try {
        const { PlatformApi } = this.libraries;
        let folder;
        let viewItems = [];

        if (graphics2dFilePath) {
          folderName = "geometryData_" + this.params.context.id;
          folder = await PlatformApi.IafFileSvc.addFolder(
            folderName,
            this.params.context._namespaces[0],
            undefined,
            this.params.context
          );
          await this.#updateFileDetailsInModel(folder._id);
        }

        if (graphics2dFilePath) {
          const viewItems2d = await this.#process2DChopped(
            graphics2dFilePath,
            folder,
            graphicsFileTitle
          );

          if (viewItems2d && viewItems2d.length > 0) {
            viewItems = viewItems.concat(viewItems2d);
          }
        }

        resolve(viewItems);
      } catch (error) {
        reject(error);
      }
    });
  }

  async #process2D(graphics2dFilePath, folder, title) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing process2D method, orch_run_id: ${this.params.orchRunId}`
        );
        if (!graphics2dFilePath) return resolve([]);

        const { ModelFileReader } = this.libraries;
        //Upload Scz
        const filePath =
          this.params.result.bimFilePath + "/" + graphics2dFilePath;
        const filename = this.params.result.filename + "_2d.scz";
        const opts = { filename };
        const tags = ["#invgraphicsfile#"];

        const uploadParams = {
          filePath,
          opts,
          tags,
          folder,
          context: this.params.context,
        };
        const uploadRes = await ModelFileReader.uploadLargeFile(uploadParams);
        let sczFileItem = {};

        if (uploadRes?.isFileuploaded) {
          await this.#cacheFile(uploadRes.file?._id);
          sczFileItem = await this.#createFileItem(uploadRes.file);
        }
        let viewItems = [];
        //Upload mapping file
        const hoops_node_mapping_file = "hoops_node_mapping2d.json";
        const mappingFilePath = this.#updatePath(
          filePath,
          hoops_node_mapping_file
        );
        const mapFileUploadParams = {
          filePath: mappingFilePath,
          opts: { filename: hoops_node_mapping_file },
          tags: [],
          folder,
          readFile: true,
          context: this.params.context,
        };
        const mappingFile = await ModelFileReader.uploadLargeFile(
          mapFileUploadParams
        );

        if (mappingFile?.isFileuploaded) {
          const mappingFileItem = await this.#createFileItem(mappingFile.file);
          const mapping2D = JSON.parse(mappingFile.rawFile);

          Object.keys(mapping2D).forEach((key) => {
            viewItems.push({
              title: key,
              aspect: "View2d",
              viewableResources: [sczFileItem._id],
              resources: [mappingFileItem._id],
            });
          });
        }

        resolve(viewItems);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at process2D method, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  async #process2DChopped(graphicsFilePath, folder, title) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing process2DChopped method arguments, orch_run_id: ${this.params.orchRunId}`
        );
        const { path, ModelFileReader } = this.libraries;
        let viewItem = {
          title: title ? title : "2dGraphics",
          aspect: "View2d",
        };
        // Upload Scz
        const filePath =
          this.params.result.bimFilePath + "/" + graphicsFilePath;
        const filePathParts = graphicsFilePath.split("/");
        const filename =
          this.params.result.filename + "_" + filePathParts[1] + ".scz";
        const opts = { filename };
        const tags = ["#invgraphicsfile#"];

        const uploadParams = {
          filePath,
          opts,
          tags,
          folder,
          context: this.params.context,
        };
        const uploadRes = await ModelFileReader.uploadLargeFile(uploadParams);
        let sczFileItem = {};

        if (uploadRes?.isFileuploaded) {
          await this.#cacheFile(uploadRes.file?._id);
          sczFileItem = await this.#createFileItem(uploadRes.file);
          viewItem.viewableResources = [sczFileItem._id];
        }
        //const dir = path.dirname(filePath)
        const hoops_node_mapping_file = "hoops_node_mapping2d.json";
        //const mappingFilePath = dir + '/' + hoops_node_mapping_file

        const mappingFilePath = this.#updatePath(
          filePath,
          hoops_node_mapping_file
        );
        const mapFileUploadParams = {
          filePath: mappingFilePath,
          opts: { filename: hoops_node_mapping_file },
          tags: [],
          folder,
          context: this.params.context,
        };
        const mappingFile = await ModelFileReader.uploadLargeFile(
          mapFileUploadParams
        );

        if (mappingFile?.isFileuploaded) {
          const mappingFileItem = await this.#createFileItem(mappingFile.file);

          viewItem.resources = [mappingFileItem._id];
        }

        resolve([viewItem]);
      } catch (error) {
        reject(error);
      }
    });
  }

  async #process3D(graphicsFilePath, folder, title, layers, modelBounding) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing process3D method, orch_run_id: ${this.params.orchRunId}`
        );
        const { path, ModelFileReader } = this.libraries;
        let viewItem = {
          title: title ? title : "3dGraphics",
          aspect: "View3d",
        };
        //Upload Scz
        const filePath =
          this.params.result.bimFilePath + "/" + graphicsFilePath;
        const filePathParts = graphicsFilePath.split("/");
        const filename =
          this.params.result.filename + "_" + filePathParts[1] + ".scz";
        const opts = { filename };
        const tags = ["#invgraphicsfile#"];

        //Scz file upload
        const uploadParams = {
          filePath,
          opts,
          tags,
          folder,
          context: this.params.context,
        };
        const uploadRes = await ModelFileReader.uploadLargeFile(uploadParams);

        if (uploadRes?.isFileuploaded) {
          await this.#cacheFile(uploadRes.file?._id);
          const sczFileItem = await this.#createFileItem(uploadRes.file);

          viewItem.viewableResources = [sczFileItem._id];
          layers && (viewItem.layers = layers);
          modelBounding && (viewItem.modelBounding = modelBounding);
        }

        //Upload mapping file
        const hoops_node_mapping_file = "hoops_node_mapping.json";
        const mappingFilePath = this.#updatePath(
          filePath,
          hoops_node_mapping_file
        );
        const mapFileUploadParams = {
          filePath: mappingFilePath,
          opts: { filename: hoops_node_mapping_file },
          tags: [],
          folder,
          context: this.params.context,
        };
        const mapFile = await ModelFileReader.uploadLargeFile(
          mapFileUploadParams
        );

        if (mapFile?.isFileuploaded) {
          const mappingFileItem = await this.#createFileItem(mapFile.file);

          viewItem.resources = [mappingFileItem._id];
        }

        resolve([viewItem]);
      } catch (error) {
        reject(error);
      }
    });
  }

  async #updateViewableResources(viewItems) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing updateViewableResources method, orch_run_id: ${this.params.orchRunId}`
        );
        console.log(`SGPK viewItems count: ${viewItems.length}`);
        if (!viewItems || viewItems.length <= 0) {
          console.log('SGPK !viewItems || viewItems.length <= 0')
          return resolve(true)
        };

        const { PlatformApi } = this.libraries;

        for (let viewItem of viewItems) {
          console.log('SGPK creating view item', viewItem)
          let resViewItem = await PlatformApi.IafItemSvc.createRelatedItems(
            this.params.result.viewcolid,
            [viewItem],
            this.params.context
          );
          console.log('SGPK created related item')

          resViewItem = resViewItem?._list[0];
          let relationShips = [];
          let relatedToIds = viewItem.viewableResources
            ? viewItem.viewableResources
            : [];

          relatedToIds = relatedToIds
            ? relatedToIds.concat(viewItem.resources ? viewItem.resources : [])
            : [];
          if (relatedToIds.length > 0) {
            const relations = {
              _relatedFromId: resViewItem._id,
              _relatedUserItemId: this.params.result.filecolid,
              _relatedToIds: relatedToIds,
            };

            relationShips.push(relations);
            await PlatformApi.IafItemSvc.addRelationsBulk(
              this.params.result.viewcolid,
              relationShips,
              this.params.context
            );
            console.log('SGPK created relationship')
          }
        }

        resolve(true);
      } catch (error) {
        reject(error);
      }
    });
  }

  async #updateFileDetailsInModel(folderId) {
    return new Promise(async (resolve, reject) => {
      try {
        console.log(
          `SGPK RefApp Executing updateFileDetailsInModel method, orch_run_id: ${this.params.orchRunId}`
        );
        const { _, ModelFileReader, PlatformApi } = this.libraries;
        //let userAttributes = {}
        let thumbnailPath =
          _.get(
            _.find(this.params.result.manifest.occurrences, (oc) => oc.data),
            "data.thumbnail"
          ) || "thumbnail.png";

        thumbnailPath = this.params.result.bimFilePath + "/" + thumbnailPath;
        const uploadParams = {
          filePath: thumbnailPath,
          _namespaces: this.params.context._namespaces[0],
          folderId,
          tags: undefined,
          ctx: this.params.context,
          opts: {},
        };
        const fileUpload = await ModelFileReader.uploadSmallFile(uploadParams);

        if (fileUpload?.isFileuploaded) {
          const { fileInfo } = fileUpload;
          const userAttributes = {
            thumbnail: {
              fileId: fileInfo[0]._id,
              fileVersionId: fileInfo[0]._tipId,
            },
            [this.params.ext]: {
              fileId: this.params.result._fileId,
              fileVersionId: this.params.result._fileVersionId,
            },
          };

          const compItem = await PlatformApi.IafItemSvc.getNamedUserItem(
            this.params.result.compositeitemid,
            this.params.context
          );
          const version = compItem._versions[0];

          if (version.hasOwnProperty("_userAttributes")) {
            Object.assign(version._userAttributes, userAttributes);
          } else {
            version._userAttributes = userAttributes;
          }

          await PlatformApi.IafItemSvc.updateNamedUserItemVersion(
            compItem._id,
            version._id,
            version,
            this.params.context
          );
        }

        resolve(true);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at updateFileDetailsInModel method, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  async #cacheFile(_fileId) {
    return new Promise(async (resolve, reject) => {
      try {
        const { PlatformApi } = this.libraries;

        await PlatformApi.IafGraphicsSvc.cacheGraphicsFiles(
          [{ _fileId }],
          this.params.context
        );
        console.log(
          `SGPK RefApp Successfully cached file_id: ${_fileId} in graphics service, orch_run_id: ${this.params.orchRunId}`
        );

        resolve(true);
      } catch (error) {
        reject(error);
      }
    });
  }

  async #createFileItem(fileObj) {
    return new Promise(async (resolve, reject) => {
      try {
        const { PlatformApi } = this.libraries;
        const fileAttr = {
          _fileId: fileObj._id,
          _fileVersionId: fileObj._tipId,
          filename: fileObj._name,
        };

        const { _list = [] } =
          (await PlatformApi.IafItemSvc.createRelatedItems(
            this.params.result.filecolid,
            [fileAttr],
            this.params.context
          )) || {};

        resolve(_list[0]);
      } catch (error) {
        console.log(
          `SGPK RefApp Error at createFileItem method, orch_run_id: ${this.params.orchRunId}`,
          error
        );

        reject(error);
      }
    });
  }

  #updatePath(filePath, fileName) {
    try {
      let splitPaths = filePath.split("/");

      splitPaths[splitPaths.length - 1] = fileName;
      const modifiedFilePath = splitPaths.join("/");

      return modifiedFilePath;
    } catch (error) {
      console.log(
        `SGPK RefApp Error at #updatePath, orch_run_id: ${this.params.orchRunId}`,
        error
      );

      throw error;
    }
  }

  #removeFileExtension(fileName) {
    const split = fileName.split(".");
    const extension = split.pop();
    const exists = extension && extension.length === 3;

    return exists ? split.join(".") : fileName;
  }
}

async function importModels(params, libraries, ctx) {
  console.log("SGPK importModels: Starting execution...");
  const { PlatformApi, IafScriptEngine } = libraries;

  async function fetchBimpkFiles() {
    console.log("SGPK importModels: Fetching BIMPK or SGPK files...");
    const criteria = { 
      _name: ".*(bimpk|sgpk)"
    };
    let fileItems;
    try {
      fileItems = await PlatformApi.IafFileSvc.getFiles(criteria, ctx);
      console.log("SGPK Fetched BIMPK or SGPK files:", JSON.stringify(fileItems));
      fileItems = fileItems._list.filter(file => 
        (file._name.endsWith('.bimpk') || file._name.endsWith('.sgpk')) 
        && file._type == 'file');
      console.log("SGPK Fetched BIMPK or SGPK files only:", JSON.stringify(fileItems));
    } catch (error) {
      console.error("SGPK Error fetching BIMPK or SGPK files:", JSON.stringify(error));
      throw error;
    }

    if (!fileItems?.length) {
      console.log("SGPK No BIMPK or SGPK files found");
      return [];
    }

    const fileData = await Promise.all(
      fileItems.map(async (file) => {
        const fileVersion = await PlatformApi.IafFile.getFileVerisons(file._id, ctx);
        return {
          file_id: file._id,
          fileVersion_id: fileVersion._list[0]._id,
        };
      })
    );

    console.log("SGPK BIMPK files found:", JSON.stringify(fileData));
    return fileData;
  }

  async function pollOrchestrator(orchResult, index) {
    const MAX_ATTEMPTS = 680; 
    let attempts = 0;

    while (attempts < MAX_ATTEMPTS) {
      try {
        const orchRunResult = await PlatformApi.IafDataSource.getOrchRunStatus(orchResult.id, ctx);
        console.log('SGPK import models orchRunResult: ', orchRunResult)
        const orchStepRunStatus = orchRunResult[0].orchrunsteps;

        const errStatus = orchStepRunStatus.filter(run_status => run_status._status === "ERROR");
        const queuedStatus = orchStepRunStatus.filter(run_status => run_status._status === "QUEUED");
        const runningStatus = orchStepRunStatus.filter(run_status => run_status._status === "RUNNING");

        console.log(`SGPK Import orchestrator ${index} status:`, {
          orchRunResult: orchRunResult,
          errors: errStatus.length,
          queued: queuedStatus.length,
          running: runningStatus.length,
          attempt: attempts
        });

        if (!_.isEmpty(errStatus)) {
          throw new Error(`SGPK Import orchestrator ${index} encountered errors: ${JSON.stringify(errStatus)}`);
        }

        if (_.isEmpty(queuedStatus) && _.isEmpty(runningStatus)) {
          orchStepRunStatus.forEach((step) => step.status = 'COMPLETED');
          return {
            orchestratorId: orchResult.id,
            status: 'COMPLETED',
            steps: orchStepRunStatus
          };
        }

        const startWait = Date.now();
        while (Date.now() - startWait < 10000) {
          const x = Math.random();
        }

        attempts++;
      } catch (error) {
        console.error(`SGPK Error polling orchestrator ${index}:`, JSON.stringify(error));
        throw error;
      }
    }

    throw new Error(`SGPK Orchestrator ${index} timed out after 60 minutes`);
  }

  async function runOrchestrators(orchestrator, bimpkFiles) {
    const task = _.find(orchestrator.orchsteps, { _sequenceno: 1 });
    const seqTypeId = task._compid;

    const runParams = bimpkFiles.map((file) => ({
      orchestratorId: orchestrator.id,
      _actualparams: [{
        sequence_type_id: seqTypeId,
        params: {
          _fileId: file.file_id,
          _fileVersionId: file.fileVersion_id
        }
      }]
    }));

    console.log("SGPK Starting orchestrator runs with params:", JSON.stringify(runParams));

    const results = await Promise.all(
      runParams.map(async (param, index) => {
        try {
          const orchResult = await IafScriptEngine.runDatasource(param, ctx);
          console.log(`SGPK Orchestrator ${index} started:`, JSON.stringify(orchResult));
          return await pollOrchestrator(orchResult, index);
        } catch (error) {
          console.error(`SGPK Error in orchestrator ${index}:`, error);
          throw error;
        }
      })
    );

    return results;
  }

  try {
    // Fetch BIMPK files
    const bimpkFiles = await fetchBimpkFiles();
    if (!bimpkFiles.length) {
      throw new Error("SGPK No BIMPK files found");
    }

    // Get orchestrator configuration
    let orchestrators = await IafScriptEngine.getDatasources(
      { _namespaces: ctx._namespaces },
      ctx
    );
    orchestrators = orchestrators.filter(orch => orch._userType === "bimpk_import");

    if (!orchestrators.length) {
      throw new Error("SGPK No valid orchestrator found");
    }

    // Run orchestrators and wait for completion
    const results = await runOrchestrators(orchestrators[0], bimpkFiles);
    console.log("SGPK All orchestrators completed:", JSON.stringify(results));

    return results;
  } catch (error) {
    console.error("SGPK Error in importModels:", JSON.stringify(error));
    throw error;
  }
}

async function importModel(params, libraries, ctx) {
  const { PlatformApi } = libraries;
  let importType = 'bimpk';
  console.log("SGPK importModel: Starting execution with params:", JSON.stringify(params));

  // Helper function to update project models
  async function updateProjectModels(result, importType, PlatformApi, ctx) {
    console.log({
      result,
      importType
    })
    const workspaces = await PlatformApi.IafWorkspace.getAll(ctx);
    const projects = workspaces.filter(ws => ws._userType === "project_workspace");
    const project = projects.sort((a, b) => b._metadata._createdAt - a._metadata._createdAt)[0];

    if (!project) {
      throw new Error("SGPK No project workspace found");
    }

    const modelBimData = {
      model: {
        model: result.bim_model_nci._id,
        modelVersionId: result.bim_model_nci._versions[0]._id,
      },
      bimpk: {
        filename: `${result.filename}.${importType}`,
        _fileId: result._fileId,
        _fileVersionId: result._fileVersionId
      }
    };

    const updatedProject = {
      ...project,
      _userAttributes: {
        ...project._userAttributes,
        currentModels: [
          ...(project._userAttributes.currentModels || []),
          modelBimData
        ]
      }
    };

    if (updatedProject._userAttributes.currentModels.length > 1) {
      updatedProject._userAttributes.multiModel = true;
    }

    return await PlatformApi.IafProj.update(updatedProject, ctx);
  }

  try {
    const helper = new Helper(params, libraries);
    const { filename, ext } = await helper.getFileMetaData(params?.actualParams._fileId);

    params.filename = filename;
    params.ext = ext?.toLowerCase();

    // Validate input
    const validateInput = new InputValidation(params, libraries, ctx);
    await validateInput.validate();

    // Process import based on file type
    let result = {};
    if (params.ext === "bimpk") {
      console.log('else bimpkImport', params)
      const bimpkImport = new BimpkImport(params, libraries, ctx);
      result = await bimpkImport.initialize();
      console.log('SGPK bimpkImport result', result)
    } else {
      importType = "sgpk"
      
      console.log(' sgpkImport params', params)
      console.log(' sgpkImport ctx', ctx)
      const sgpkImport = new SgpkImport(params, libraries, ctx);
      result = await sgpkImport.initialize();
      console.log('SGPK SgpkImport result', result)
    }

    //Process SCZ import
    params.result = result;
    const sczImport = new SczTarget(params, libraries, ctx);
    const sczImportRes = await sczImport.intialize();
    console.log("SGPK SCZ import completed:", sczImportRes);

    //Update project with new model
    const updatedProject = await updateProjectModels(result, importType, PlatformApi, ctx);
    console.log("SGPK Project updated with new model:", JSON.stringify(updatedProject));

    //Prepare final response
    const response = {
      filecolid: result.filecolid,
      viewcolid: result.viewcolid,
      compositeitemid: result.compositeitemid,
    };

    if (result.myCollections) {
      response.myCollections = result.myCollections;
    }

    console.log('SGPK importModel res: ', response)

    return response;
   
  } catch (error) {
    console.error("SGPK Error in importModel:", error);
    throw error;
  }
}

const cacheSourceFileGraphicsIds = async (params, libraries, ctx) => {
  return new Promise(async (resolve, reject) => {
    try {
      const { IafScriptEngine } = libraries;
      const { model_els_coll, data_cache_coll } = params.inparams.myCollections;

      console.log("SGPK --> RefApp cache elems: " + model_els_coll._name);
      console.log("SGPK --> RefApp cache data: " + data_cache_coll._name);

      const { _list: sourcefileList = [] } =
        (await IafScriptEngine.getDistinct(
          {
            collectionDesc: {
              _userType: model_els_coll._userType,
              _userItemId: model_els_coll._userItemId,
            },
            field: "source_filename",
            options: { getCollInfo: true },
          },
          ctx
        )) || {};
      const sourcefileNames =
        sourcefileList[0]?._versions[0]?._relatedItems?.source_filename;
      let cacheDataItems = [];

      for (let i = 0; i < sourcefileNames.length; i++) {
        const { _list: packageIdList = [] } =
          (await IafScriptEngine.getDistinct(
            {
              collectionDesc: {
                _userType: model_els_coll._userType,
                _userItemId: model_els_coll._userItemId,
              },
              query: { source_filename: sourcefileNames[i] },
              field: "package_id",
              options: { getCollInfo: true },
            },
            ctx
          )) || {};

        cacheDataItems.push({
          dataType: "sourcefileToPkgIds",
          data: {
            source_filename: sourcefileNames[i],
            package_id:
              packageIdList[0]?._versions[0]?._relatedItems?.package_id,
          },
        });
      }
      const bim_els = await IafScriptEngine.createItemsBulk(
        {
          _userItemId: data_cache_coll._userItemId,
          _namespaces: ctx._namespaces,
          items: cacheDataItems,
        },
        ctx
      );

      await IafScriptEngine.setVar("outparams", { cacheDataItems: bim_els });

      console.log("SGPK RefApp Create Cache Data: source filenames to package_ids");

      resolve(true);
    } catch (err) {
      console.log("SGPK RefApp Error at cacheSourceFileGraphicsIds");
      console.error(err);

      reject(err);
    }
  });
};

async function createModelDataCache(params, libraries, ctx) {
  return new Promise(async (resolve, reject) => {
    try {
      const { IafScriptEngine } = libraries;

      await cacheSourceFileGraphicsIds(params, libraries, ctx);
      const outParams = await IafScriptEngine.getVar("outparams");

      resolve(outParams);
    } catch (error) {
      console.log("SGPK RefApp Error at createModelDataCache");
      console.error(error);

      reject(error);
    }
  });
}