const mapBox = {
  async createMapboxConfigColl(PlatformApi, ctx) {
    const { IafItemSvc } = PlatformApi;
    const mapBoxCollDef = {
      _name: `MapBox Configs Collection`,
      _shortName: "mapbox_config_coll",
      _description: "Named User Collection to Mapbox temp token configs",
      _namespaces: ctx._namespaces,
      _userType: "mapbox_config_coll",
      _encryptionEnabled: true
    };
    let mapBoxConfigColl = await IafItemSvc.createNamedUserItems(
      [mapBoxCollDef],
      'NamedUserCollection',
      ctx
    );
    console.log('mapBoxConfigColl', mapBoxConfigColl);
    return mapBoxConfigColl._list[0];
  },

  async checkForMapconfigColl(PlatformApi, ctx) {
    const { IafItemSvc } = PlatformApi;
    let mapboxCollRes = await IafItemSvc.getNamedUserItems(
      { "query": { _userType: 'mapbox_config_coll' } },
      ctx,
      {}
    );
    mapboxCollRes = mapboxCollRes._list;
    let mapboxColl;
    if (!mapboxCollRes.length) {
      console.log('No Mapbox config collection')
      mapboxColl = await this.createMapboxConfigColl(PlatformApi, ctx);
    } else {
      console.log('Mapbox config collection found')
      mapboxColl = mapboxCollRes[0] 
    }
    console.log('mapboxColl', mapboxColl);
    return mapboxColl
  },

  async addSecretToMapboxColl(username, scopes, expiry, secret, coll, projId, PlatformApi, ctx) {
    const { IafItemSvc } = PlatformApi;
		const mapboxRelItems = [
			{
        '_projId': projId,
        '_username': username, 
        '_scopes': scopes, 
        '_expiry': expiry,
        '._secret': secret
			}
		];
    const mapboxItems = await IafItemSvc.createRelatedItems(
			coll._id,
			mapboxRelItems,
			ctx
		);
    console.log('mapboxItems', mapboxItems);
    return mapboxItems;
  },

  async checkMapboxOrch(PlatformApi, IafScriptEngine, ctx) {
    const { IafDataSource } = PlatformApi;
    const allDatasources = await IafDataSource.getOrchestrators();
    console.log('allDatasources', allDatasources)
    let mapboxTokenOrch = allDatasources._list.find((item) => 
      item._userType === "mapbox_temp_token_orch"
    );
    console.log('mapboxTokenOrch', mapboxTokenOrch)
    if (!mapboxTokenOrch) {
      mapboxTokenOrch = await this.createMapboxOrch(PlatformApi, ctx);
      console.log('added mapboxTokenOrch', mapboxTokenOrch)
    } 
    console.log('mapboxTokenOrch._permissionprofileid', mapboxTokenOrch._permissionprofileidoxTokenOrch)
    if (mapboxTokenOrch && !mapboxTokenOrch._permissionprofileid) {
      console.log('No perm prof id', mapboxTokenOrch._permissionprofileid)
      mapboxTokenOrch = await this.addPermProfileId(mapboxTokenOrch, PlatformApi, IafScriptEngine, ctx);
      console.log('added perm profile id to orch: ', mapboxTokenOrch);
    }
    console.log('returning mapboxTokenOrch', mapboxTokenOrch)
    return mapboxTokenOrch;
  },

  async createPermProfIfNonExists(PlatformApi, ctx) {
    const { IafProj } = PlatformApi;
    const project = IafProj.getCurrent();
    let permProfile;
    let permCheck = await this.checkPermProfile(PlatformApi, ctx);
    console.log('permCheck', permCheck)
    if (permCheck) {
      permProfile = permCheck;
    } else {
      const coll = await this.checkForMapconfigColl(PlatformApi, ctx);
      permProfile = await this.createMapboxPermProfile(project, coll, PlatformApi);
    }
    console.log('createPermProfIfNonExists return', permProfile)
    return permProfile
  },

  async addPermProfileId(mapboxTokenOrch, PlatformApi, IafScriptEngine, ctx) {
    console.log('addPermProfileId mapboxTokenOrch: ', mapboxTokenOrch)
    const { IafDataSource } = PlatformApi;
    const permProfile = await this.createPermProfIfNonExists(PlatformApi, ctx);
    console.log('addPermProfileId permProfile: ', permProfile);
    let updatedOrchDef = mapboxTokenOrch;
    updatedOrchDef._permissionprofileid = permProfile?._id;
    console.log('addPermProfileId updatedOrchDef: ', updatedOrchDef);
    try {
      const updatedOrch = await IafDataSource.updateOrchestrator(updatedOrchDef.id, updatedOrchDef, ctx);
      console.log('addPermProfileId updatedOrch: ', updatedOrch);
      return updatedOrch;
    } catch (e) {
      console.log('addPermProfileId error', e)
    }
  },

  async createMapboxOrch(PlatformApi, ctx) {
    const { IafDataSource, IafProj } = PlatformApi;
    const project = await IafProj.getCurrent();
    const permProfile = await this.createPermProfIfNonExists(PlatformApi, ctx);
    console.log('permProfile at createMapboxOrch', permProfile)

	  const mapboxTokenDsResult = await IafDataSource.createOrchestrator(
	    {
	      _name: "Request MapBox Token",
	      _description: "Requests a MapBox token for a user",
	      _namespaces: project._namespaces,
	      _userType: "mapbox_temp_token_orch",
	      _schemaversion: "2.0",
        _instant: true,
        _permissionprofileid: permProfile._id,
	      _params: {
	        tasks: [
	          {
	            name: "default_script_target",
	            _actualparams: {
	              userType: "iaf_mapbox",
	              _scriptName: "requestMapboxToken"
	            },
	            _sequenceno: 1,
	          },
	        ],
	      },
	    },
	    ctx
	  );
    console.log('mapboxTokenDsResult: ', mapboxTokenDsResult)
    return mapboxTokenDsResult
  },

  async runMapboxOrch(projId, PlatformApi, IafScriptEngine, ctx) {
    const { IafDataSource } = PlatformApi;
    const datasources = await IafDataSource.getOrchestrators(null, ctx, null);
    console.log("datasources: ", datasources); 

    const mapboxOrch = datasources._list.find((item) => 
      item._userType === "mapbox_temp_token_orch"
    )
    console.log("mapboxOrch: ", mapboxOrch);

    const task = _.find(mapboxOrch.orchsteps, { _sequenceno: 1 });
    const seqTypeId = task._compid;

    try {
      const orchRes = await IafScriptEngine.runDatasource(
        {
          orchestratorId: mapboxOrch.id,
          _actualparams: [{
            sequence_type_id: seqTypeId,
            params: {
              projId: projId
            }
          }],
        }, 
        ctx
      );
      console.log('orchRes', orchRes)

      return orchRes;
    } catch (e) {
      console.log('runMapboxOrch err: ', e)
      throw e
    }
  },

  async createMapboxPermProfile(project, coll, PlatformApi) {
    console.log('createMapboxPermProfile coll', coll)
    const { IafPassSvc } = PlatformApi;
    const workspaceId = project._id;
    const irnValue = "passportsvc:workspace:" + workspaceId;
    console.log('running createMapboxPermProfile');
    const ctx = { _namespaces: project._namespaces };
    let permProfiles = [
      {
        _name: "Mapbox token orchestrator perms",
        _userType: "mapbox_token_perm",
        _namespaces: [project._namespaces[0]],
        _permissions: [
          {
            _actions: ["READ", "SHARE"],
            _namespace: project._namespaces[0],
            _resourceDesc: {
              _irn: irnValue,
            },
          },
          {
            _actions: ["READ"],
            _namespace: project._namespaces[0],
            _resourceDesc: {
              _irn: `itemsvc:nameduseritem:${coll._id}`,
            },
          }
        ]
      }
    ];
    console.log("createMapboxPermProfile permProfiles:", permProfiles)
    try {
      let permProfile = await IafPassSvc.createPermissionProfiles(
        permProfiles,
        ctx
      );
      let permProfReq;
      let i = 0;
      while (!permProfReq) {
        try {
          console.log(`checking for created perm, attempt ${i + 1}`)
          permProfReq = await this.checkPermProfile(PlatformApi, ctx);
        } catch (error) {
          console.error("Error occurred during permission profile check:", error);
        }
      }
      console.log("createMapboxPermProfile permProfile after check:", permProfReq)
      return permProfReq;
    } catch (e) {
      console.log('createMapboxPermProfile error: ', e)
    }
  },

  async checkPermProfile(PlatformApi, ctx) {
    const { IafPassSvc } = PlatformApi;
    console.log('running checkPermProfile');
    const permissionProfiles = await IafPassSvc.getPermissionProfiles(
      { },
      ctx,
      { _pageSize: 25, _offset: 0 }
    );
    console.log("permissionProfiles", permissionProfiles);
    const mapboxPermProfile = permissionProfiles._list.find((item) => 
      item._userType === "mapbox_token_perm"
    );
    console.log("mapboxPermProfile", mapboxPermProfile);
    return mapboxPermProfile;
  },

  // async getMapboxToken(project, PlatformApi, IafScriptEngine, ctx) {
  //   const scriptCheck = await this.checkTokenScriptExists(PlatformApi, ctx);
  //   !scriptCheck && await this.addTokenScripts(PlatformApi, ctx);
  //   let orchCheck = await this.checkMapboxOrch(PlatformApi, IafScriptEngine, ctx);
  //   !orchCheck && await this.createMapboxOrch(PlatformApi, ctx);
  //   const orchRes = await this.runMapboxOrch(
  //     project._id, 
  //     PlatformApi, 
  //     IafScriptEngine, 
  //     ctx
  //   );
  //   return orchRes?._result?.token;    
  // },

  async getMapboxToken(project, PlatformApi, IafScriptEngine, ctx) {
    // Start logging
    console.log('Starting mapbox token retrieval process for project:', project._id);

    const collCheck = await this.checkForMapconfigColl(PlatformApi, ctx);
    console.log('Checked for mapbox config collection:', collCheck);
    const scriptCheck = await this.checkTokenScriptExists(PlatformApi, ctx);
    console.log(`Checked if token script exists: ${scriptCheck}`);

    if (!scriptCheck) {
        console.log('Adding token scripts...');
        await this.addTokenScripts(PlatformApi, ctx);
    }

    let orchCheck = await this.checkMapboxOrch(PlatformApi, IafScriptEngine, ctx);
    console.log(`Checked if mapbox orchestration exists: ${JSON.stringify(orchCheck)}`);

    const orchRes = await this.runMapboxOrch(project._id, PlatformApi, IafScriptEngine, ctx);
    console.log(`Orchestration result:`, orchRes);

    if (orchRes && orchRes._result?.token) {
        console.log('Token retrieved successfully:', orchRes._result.token);
    } else {
        console.log('Failed to retrieve token from orchestration result.');
    }

    return orchRes?._result?.token;
  },

  checkTokenFreshness(token) {
    try {
      const tokenBase64 = token.split('.')[1];
      const decodedToken = JSON.parse(atob(tokenBase64));
      const exp = decodedToken.exp; 
      const now = Math.floor(Date.now() / 1000);
      if (exp < now) {
        console.log("Token has expired. Requesting a new one...");
        return false;
      } else {
        console.log("Mapbox token valid");
        return true;
      }
    } catch (e) {
      console.error('Error decoding token:', e);
      return false;
    }
  },

  async checkTokenScriptExists(PlatformApi, ctx) {
    try {
      console.log('checking if iaf_mapbox script exists')
      const { IafScripts } = PlatformApi;
      const scripts = await IafScripts.getScripts({}, ctx);
      const tokenScript = _.find(scripts, { _userType: "iaf_mapbox" });
      if (tokenScript) {
        console.log('iaf_mapbox tokenScript exists', tokenScript)
        return true;
      } else {
        console.log('iaf_mapbox script does not exist')
        return false;
      }
    } catch (e) {
      console.log('Error checkTokenScriptExists:', e);
      return false;
    }
  },

  async addTokenScripts(PlatformApi, ctx) {
    console.log('running addTokenScripts')
    const { IafItemSvc } = PlatformApi;
    const scriptContent = `
      const MapboxScopes = {
        // BASIC READ SCOPES
        DATASETS_READ: "datasets:read",
        FONTS_READ: "fonts:read",
        STYLES_READ: "styles:read",
        TILES_READ: "tiles:read",

        // BROADER READ SCOPES
        MAP_READ: "map:read",
        TOKENS_READ: "tokens:read",
        UPLOADS_READ: "uploads:read",
        VISION_READ: "vision:read",

        // WRITE SCOPES
        DATASETS_WRITE: "datasets:write",
        STYLES_WRITE: "styles:write",
        TILES_WRITE: "tiles:write",

        TOKENS_WRITE: "tokens:write",
        UPLOADS_WRITE: "uploads:write",
        MAP_WRITE: "map:write",
        VISION_WRITE: "vision:write"
      };

      async function requestMapboxToken(input, libraries, ctx) {
        const { IafScriptEngine } = libraries;
        const mapboxConfigQuery = {
          parent: {
            collectionDesc: {
              _userType: "mapbox_config_coll",
            },
            options: {
              page: {
                getAllItems: true,
              },
              sort: {
                '_metadata._createdAt': -1
              }
            },
            query: {
              _projId: input.actualParams.projId
            },
          },
        };

        const mapboxConfigs = await IafScriptEngine.findWithRelated(
          mapboxConfigQuery, ctx
        );
        const latestConfig = mapboxConfigs._list[0];
        const secret = latestConfig['._secret'];
        const username = latestConfig._username;
        const expiry = latestConfig._expiry;
        const scopes = latestConfig._scopes;

        return await generateTemporaryMapboxToken(
          secret, 
          scopes, 
          expiry, 
          username
        );
      }

      async function generateTemporaryMapboxToken(
        secret, 
        scopes = [
          MapboxScopes.STYLES_READ,
          MapboxScopes.DATASETS_READ,
          MapboxScopes.MAP_READ,
          MapboxScopes.FONTS_READ,    
          MapboxScopes.TOKENS_WRITE          
        ], 
        expiresInSeconds = 3600, 
        username
      ) {
        const MAPBOX_SECRET_TOKEN = secret;
        const MAPBOX_USERNAME = username;
        const url = \`https://api.mapbox.com/tokens/v2/\${MAPBOX_USERNAME}\`;
        const expires = new Date(new Date().getTime() + expiresInSeconds * 1000).toISOString();

        const requestBody = {
          expires,
          scopes,
          note: "Temporary token for IafViewer session"
        };
        const headers = {
          Authorization: \`Bearer \${MAPBOX_SECRET_TOKEN}\`,
          "Content-Type": "application/json"
        };

        try {
          return await fetch(url, {
            method: "POST",
            headers: headers,
            body: JSON.stringify(requestBody)
          });
        } catch (err) {
          console.error("Temp token Error:", JSON.stringify(err));
          return null;
        }
      }
    `;

    let scripts = [{
      _name: "Mapbox Scripts",
      _shortName: "iaf_mapbox",
      _description: "Mapbox token request api",
      _namespaces: ctx._namespaces,
      _userType: "iaf_mapbox",
      _itemClass: "Script", 
      _version: {
        _userData: scriptContent
      },
    }];

    try {
      const createdScripts = await IafItemSvc.createNamedUserItems(
        scripts, 
        "Script", 
        ctx
      );
      console.log('created mapbox script: ', createdScripts); 
    } catch (e) {
      console.error('Error addTokenScripts:', e);
    }
  }
};

export default mapBox;