const getModelVersions = async (param, PlatformApi, ctx) => {
  console.log(JSON.stringify({ message: "Fetching Models" }));
  try {
    let bim_models = await PlatformApi.IafScriptEngine.getCompositeCollections(
      {
        query: {
          _userType: "bim_model_version",
          _namespaces: {
            $in: ctx._namespaces,
          },
          _itemClass: "NamedCompositeItem",
        },
      },
      ctx,
      { getLatestVersion: true }
    );
    console.log("found Models", JSON.stringify(bim_models));
    return bim_models._list;
  } catch (e){
    console.log("getModelVersions error: ", JSON.stringify(e))
  }

};

const mapElementsType = async (param, PlatformApi, ctx) => {
  const modelRelatedCollection =
    await PlatformApi.IafScriptEngine.getCollectionsInComposite(
      PlatformApi.IafScriptEngine.getVar("bim_model")._id,
      null,
      ctx
    );
  const model_type_el_coll = modelRelatedCollection.find(
    (x) => x._userType === "rvt_type_elements"
  );
  const model_els_coll = modelRelatedCollection.find(
    (x) => x._userType === "rvt_elements"
  );
  let typeElements = await PlatformApi.IafScriptEngine.getItems(
    {
      _userItemId: model_type_el_coll._userItemId,
      options: {
        page: {
          getAllItems: true,
        },
      },
    },
    ctx
  );
  let assetTypeMap = await PlatformApi.IafScriptEngine.getItems(
    {
      collectionDesc: {
        _userType: "iaf_dt_type_map_defs_coll",
        _namespaces: ctx._namespaces,
      },
      options: {
        page: {
          getAllItems: true,
        },
      },
    },
    ctx
  );
  let elements = await PlatformApi.IafScriptEngine.getItems(
    {
      _userItemId: model_els_coll._userItemId,
      options: {
        page: {
          getAllItems: true,
        },
      },
    },
    ctx
  );
  console.log(JSON.stringify({ message: "Re-mapping Type Elements" }));
  for (let typeElement of typeElements) {
    if (
      typeElement.properties.hasOwnProperty("Revit Family") &&
      typeElement.properties.hasOwnProperty("Revit Type")
    ) {
      let _myRow = assetTypeMap.find(
        (x) =>
          x["Revit Family"] == typeElement.properties["Revit Family"].val &&
          x["Revit Type"] == typeElement.properties["Revit Type"].val
      );
      if (_myRow) {
        if (_myRow.hasOwnProperty("ElementCategory")) {
          typeElement.ElementCategory = _myRow.ElementCategory;
        }
        if (_myRow.hasOwnProperty("ElementType")) {
          typeElement.ElementType = _myRow.ElementType;
        }
      }
    }
  }
  console.log(JSON.stringify({ message: "Re-mapping Model Elements" }));
  for (let element of elements) {
    let _myVal = typeElements.find((x) => x.id == element.type_id);
    if (_myVal) {
      if (_myVal.hasOwnProperty("ElementCategory")) {
        element.ElementCategory = _myVal.ElementCategory;
      }
      if (_myVal.hasOwnProperty("ElementType")) {
        element.ElementType = _myVal.ElementType;
      }
      if (_myVal.hasOwnProperty("baType")) {
        element.baType = _myVal.baType;
      }
    }
  }
  await PlatformApi.IafScriptEngine.updateItemsBulk(
    {
      _userItemId: model_els_coll._userItemId,
      items: elements,
    },
    ctx
  );
  await PlatformApi.IafScriptEngine.updateItemsBulk(
    {
      _userItemId: model_type_el_coll._userItemId,
      items: typeElements,
    },
    ctx
  );
};

async function mapAssetCollection(params, PlatformApi, ctx) {
  await getModelVersion(params, PlatformApi, ctx);
  await mapElementsType(params, PlatformApi, ctx);
  let assetCollection = await PlatformApi.IafScriptEngine.getCollection(
    {
      _userType: "iaf_ext_asset_coll",
      _shortName: "asset_coll",
      _itemClass: "NamedUserCollection",
    },
    ctx
  );

  let assets = await PlatformApi.IafScriptEngine.getItems(
    {
      _userItemId: assetCollection._userItemId,
      options: {
        page: {
          getAllItems: true,
        },
      },
    },
    ctx
  );
  let assetTypeMap = await PlatformApi.IafScriptEngine.getItems(
    {
      collectionDesc: {
        _userType: "iaf_dt_type_map_defs_coll",
        _namespaces: ctx._namespaces,
      },
      options: {
        page: {
          getAllItems: true,
        },
      },
    },
    ctx
  );
  console.log(JSON.stringify({ message: "Re-mapping Assets" }));
  for (let asset of assets) {
    if (
      asset.properties.hasOwnProperty("Revit Family") &&
      asset.properties.hasOwnProperty("Revit Type")
    ) {
      let _myRow = assetTypeMap.find(
        (x) =>
          x["Revit Family"] == asset.properties["Revit Family"].val &&
          x["Revit Type"] == asset.properties["Revit Type"].val
      );
      if (_myRow) {
        asset.properties.ElementCategory.val = _myRow.ElementCategory;
        asset.propertiesElementType.val = _myRowElementType;
      }
    }
  }

  await PlatformApi.IafScriptEngine.updateItemsBulk(
    {
      _userItemId: assetCollection._userItemId,
      items: assets,
    },
    ctx
  );

  return true;
}


async function mapRevitTypeCollections(params, libraries, ctx) {
  const { PlatformApi, IafScriptEngine } = libraries;
  console.log("running mapRevitTypeCollections: ");
  //get models
  const getModels = async () => {
    let models = await IafScriptEngine.getCompositeCollections(
      {
        query: {
          _userType: "bim_model_version",
          _namespaces: {
            $in: ctx._namespaces,
          },
          _itemClass: "NamedCompositeItem",
        },
      },
      ctx,
      { getLatestVersion: true }
    );
    models = models._list;
    console.log(
      "Started createIndex.",
      "Create index ==== getCompositeCollection.",
      models
    );
    return models;
  }

  async function pollOrchestrator(orchResult, index) {
    const MAX_ATTEMPTS = 680;
    let attempts = 0;

    while (attempts < MAX_ATTEMPTS) {
      try {
        const orchRunResult = await PlatformApi.IafDataSource.getOrchRunStatus(orchResult.id, ctx);
        console.log('mapping orchRunResult: ', orchRunResult)
        const orchStepRunStatus = orchRunResult[0].orchrunsteps;

        const errStatus = orchStepRunStatus.filter(run_status => run_status._status === "ERROR");
        const queuedStatus = orchStepRunStatus.filter(run_status => run_status._status === "QUEUED");
        const runningStatus = orchStepRunStatus.filter(run_status => run_status._status === "RUNNING");

        console.log(`Orchestrator ${index} status:`, {
          errors: errStatus.length,
          queued: queuedStatus.length,
          running: runningStatus.length,
          attempt: attempts
        });

        if (!_.isEmpty(errStatus)) {
          throw new Error(`Orchestrator ${index} encountered errors: ${JSON.stringify(errStatus)}`);
        }

        if (_.isEmpty(queuedStatus) && _.isEmpty(runningStatus)) {
          orchStepRunStatus.forEach((step) => step.status = 'COMPLETED');
          return {
            orchestratorId: orchResult.id,
            status: 'COMPLETED',
            steps: orchStepRunStatus
          };
        }

        const startWait = Date.now();
        while (Date.now() - startWait < 10000) {
          const x = Math.random(); 
        }

        attempts++;
      } catch (error) {
        console.error(`Error polling orchestrator ${index}:`, JSON.stringify(error));
        throw error;
      }
    }

    throw new Error(`Orchestrator ${index} timed out after 30 minutes`);
  }

  async function runOrchestrators(runParams) {

    const results = await Promise.all(
      runParams.map(async (param, index) => {
        try {
          const orchResult = await IafScriptEngine.runDatasource(param, ctx);
          console.log(`Orchestrator ${index} started:`, JSON.stringify(orchResult));
          return await pollOrchestrator(orchResult, index);
        } catch (error) {
          console.error(`Error in orchestrator ${index}:`, error);
          throw error;
        }
      })
    );

    return results;
  }

  try {
    console.log(
      "mapRevitTypeCollections: Fetching models..."
    );
    const bimpkModels = await getModels();
    if (!bimpkModels.length) throw new Error(
      "mapRevitTypeCollections: No models found."
    );
    console.log(
      "mapRevitTypeCollections: Fetching orchestrator..."
    );
    let orchestrators = await IafScriptEngine.getDatasources(
      { _namespaces: ctx._namespaces },
      ctx
    );
    console.log(
      "mapRevitTypeCollections: All Orchestrators:", JSON.stringify(orchestrators)
    );
    orchestrators = _.filter(orchestrators, (orch) =>  orch._userType == "map_revit_type");
    let orchestrator = orchestrators[0];
    console.log(
      "mapRevitTypeCollections: Orchestrator retrieved:", JSON.stringify(orchestrator)
    );

    const task = _.find(orchestrator.orchsteps, { _sequenceno: 1 });
    const seqTypeId = task._compid;
    const runParams = bimpkModels.map((bm) => ({
      orchestratorId: orchestrator.id,
      _actualparams: [
        {
          sequence_type_id: seqTypeId,
          params: {
            bimModel: bm
          },
        },
      ],
    }));
    console.log("mapRevitTypeCollections runParams: ", JSON.stringify(runParams));

    const orchResults = await runOrchestrators(runParams);
    console.log("mapRevitTypeCollections orchResults: ", JSON.stringify(orchResults));

    return true;
  } catch (error) {
    console.error("mapRevitTypeCollections error:", error);
    throw error;
  }
}

async function mapRevitTypeCollection (param, PlatformApi, ctx) {
  console.log("running mapRevitTypeCollection: ");
  return new Promise(async (resolve, reject) => {
    try {
      const fetched_bim_model = param.actualParams.bimModel;
      console.log("mapRevitTypeCollection fetched_bim_model:", JSON.stringify(fetched_bim_model));
      const modelRelatedCollection =
        await PlatformApi.IafScriptEngine.getCollectionsInComposite(
          fetched_bim_model._id,
          null,
          ctx
        );
      console.log("mapRevitTypeCollection modelRelatedCollection:", JSON.stringify(modelRelatedCollection));
      const model_type_el_coll = modelRelatedCollection.find(
        (x) => x._userType === "rvt_type_elements"
      );
      console.log(
        "mapRevitTypeCollection ~ model_type_el_coll",
        JSON.stringify(model_type_el_coll)
      );
      const model_els_coll = modelRelatedCollection.find(
        (x) => x._userType === "rvt_elements"
      );
      console.log(
        "mapRevitTypeCollection ~ model_els_coll",
        JSON.stringify(model_els_coll)
      );
      let typeElements = await PlatformApi.IafScriptEngine.getItems(
        {
          _userItemId: model_type_el_coll._userItemId,
          options: {
            page: {
              getAllItems: true,
            },
          },
        },
        ctx
      );
      console.log(
        "🚀 ~ file: iaf_map_elms_type.js ~ mapRevitTypeCollection ~ typeElements",
        JSON.stringify(typeElements)
      );
      let revitTypeMap = await PlatformApi.IafScriptEngine.getItems(
        {
          collectionDesc: {
            _userType: "iaf_ref_type_map_defs_coll",
            _namespaces: ctx._namespaces,
            _name: fetched_bim_model._name
          },
          options: {
            page: {
              getAllItems: true,
            },
          },
        },
        ctx
      );
      console.log(
        "🚀 ~ file: iaf_map_elms_type.js ~ mapRevitTypeCollection ~ revitTypeMap",
        JSON.stringify(revitTypeMap)
      );
      let elements = await PlatformApi.IafScriptEngine.getItems(
        {
          _userItemId: model_els_coll._userItemId,
          options: {
            page: {
              getAllItems: true,
            },
          },
        },
        ctx
      );
      console.log(
        "🚀 ~ file: iaf_map_elms_type.js ~ mapRevitTypeCollection ~ elements before map",
        JSON.stringify(elements)
      );
      console.log(JSON.stringify({ message: "Re-mapping Type Elements" }));

      for (const typeElement of typeElements) {
        let revitFamily, revitType, ifcTypeObject, civil3dClass, autoCADClass;
        if ("Revit Family" in typeElement.properties) {
          revitFamily = typeElement.properties["Revit Family"];
        }

        if ("Revit Type" in typeElement.properties) {
          revitType = typeElement.properties["Revit Type"];
        }

        if ("IFCTypeObject" in typeElement.properties) {
          ifcTypeObject = typeElement.properties["IFCTypeObject"];
        }

        if ("Civil3D Class" in typeElement.properties) {
          civil3dClass = typeElement.properties["Civil3D Class"];
        }

        if ("AutoCAD Class" in typeElement.properties) {
          autoCADClass = typeElement.properties["AutoCAD Class"];
        }

        function updateTypeElement(_myRow) {
          if (_myRow) {
            if (_myRow.hasOwnProperty("ElementCategory")) {
              typeElement.ElementCategory = _myRow.ElementCategory;
            }
            if (_myRow.hasOwnProperty("ElementType")) {
              typeElement.ElementType = _myRow.ElementType;
            }
          }
        }

        let _myRow;

        if (
          (revitFamily && revitType) ||
          ifcTypeObject ||
          civil3dClass ||
          autoCADClass
        ) {
          if (revitFamily && revitType) {
            _myRow = revitTypeMap.find(
              (x) =>
                x["Revit Family"] === revitFamily.val &&
                x["Revit Type"] === revitType.val
            );
          } else if (ifcTypeObject) {
            _myRow = revitTypeMap.find(
              (x) => x["IFCTypeObject"] === ifcTypeObject.val
            );
          } else if (civil3dClass) {
            _myRow = revitTypeMap.find(
              (x) => x["Revit Family"] === civil3dClass.val
            );
          } else if (autoCADClass) {
            _myRow = revitTypeMap.find(
              (x) => x["AutoCAD Class"] === autoCADClass.val
            );
          }

          updateTypeElement(_myRow);
        }
      }

      console.log(JSON.stringify({ message: "Re-mapping Model Elements" }));
      for (let element of elements) {
        let _myVal = typeElements.find((x) => x.id == element.type_id);
        if (_myVal) {
          if (_myVal.hasOwnProperty("ElementCategory")) {
            element.ElementCategory = _myVal.ElementCategory;
          }
          if (_myVal.hasOwnProperty("ElementType")) {
            element.ElementType = _myVal.ElementType;
          }
        }
      }

      console.log(
        "🚀 ~ file: iaf_map_elms_type.js ~ mapRevitTypeCollection ~ elements after map",
        JSON.stringify(elements)
      );
      console.log(
        "🚀 ~ file: iaf_map_elms_type.js ~ mapRevitTypeCollection ~ typeElements after map",
        JSON.stringify(typeElements)
      );
      const updatedElems = await PlatformApi.IafScriptEngine.updateItemsBulk({
          _userItemId: model_els_coll._userItemId,
          items: elements,
        },
        ctx
      );
      console.log(
        "🚀 ~ file: iaf_map_elms_type.js ~ mapRevitTypeCollection ~ elements after update",
        JSON.stringify(updatedElems)
      );

      const updatedTypeElems = await PlatformApi.IafScriptEngine.updateItemsBulk({
          _userItemId: model_type_el_coll._userItemId,
          items: typeElements,
        },
        ctx
      );
      console.log(
        "🚀 ~ file: iaf_map_elms_type.js ~ mapRevitTypeCollection ~ typeElements after update",
        JSON.stringify(updatedTypeElems)
      );
      resolve(true);
    } catch (error) {
      console.log(JSON.stringify('mapRevitTypeCollection error', JSON.stringify(error)))
      reject(error);
    }
  });
}
