const MapboxScopes = {
  // BASIC READ SCOPES
  DATASETS_READ: "datasets:read",
  FONTS_READ: "fonts:read",
  STYLES_READ: "styles:read",
  TILES_READ: "tiles:read",

  // BROADER READ SCOPES
  MAP_READ: "map:read",
  TOKENS_READ: "tokens:read",
  UPLOADS_READ: "uploads:read",
  VISION_READ: "vision:read",

  // WRITE SCOPES
  DATASETS_WRITE: "datasets:write",
  STYLES_WRITE: "styles:write",
  TILES_WRITE: "tiles:write",

  TOKENS_WRITE: "tokens:write",
  UPLOADS_WRITE: "uploads:write",
  MAP_WRITE: "map:write",
  VISION_WRITE: "vision:write"
};

async function requestMapboxToken(input, libraries, ctx) {
  const { IafScriptEngine } = libraries;
  const mapboxConfigQuery = {
    parent: {
      collectionDesc: {
        _userType: "mapbox_config_coll",
      },
      options: {
        page: {
          getAllItems: true,
        },
        sort: {
          '_metadata._createdAt': -1
        }
      },
      query: {
        _projId: input.actualParams.projId
      },
    },
  };

  const mapboxConfigs = await IafScriptEngine.findWithRelated(
    mapboxConfigQuery, ctx
  );
  console.log('mapboxConfigs: ', mapboxConfigs);
  const latestConfig = mapboxConfigs._list[0];
  const secret = latestConfig['._secret'];
  const username = latestConfig._username;
  const expiry = latestConfig._expiry;
  const scopes = latestConfig._scopes;

  return await generateTemporaryMapboxToken(
    secret, 
    scopes, 
    expiry, 
    username
  );
}

async function generateTemporaryMapboxToken(
  secret, 
  scopes = [
    MapboxScopes.STYLES_READ,
    MapboxScopes.DATASETS_READ,
    MapboxScopes.MAP_READ,
    MapboxScopes.FONTS_READ,    
    MapboxScopes.TOKENS_WRITE          
  ], 
  expiresInSeconds = 3600, 
  username
) {
  const MAPBOX_SECRET_TOKEN = secret;
  const MAPBOX_USERNAME = username;
  const url = `https://api.mapbox.com/tokens/v2/${MAPBOX_USERNAME}`;
  const expires = new Date(new Date().getTime() + expiresInSeconds * 1000).toISOString();

  const requestBody = {
    expires,
    scopes,
    note: "Temporary token for IafViewer session"
  };
  const headers = {
    Authorization: `Bearer ${MAPBOX_SECRET_TOKEN}`,
    "Content-Type": "application/json"
  };

  try {
    const res =  await fetch(url, {
      method: "POST",
      headers: headers,
      body: JSON.stringify(requestBody)
    });
    console.log("Temp token response status:", res);
    return res
  } catch (err) {
    console.error("Temp token Error:", JSON.stringify(err));
    return null;
  }
}