const isEmpty = (ele) => {
  if (ele === undefined) {
    return true;
  } else if (ele === null) {
    return true;
  }
  return false;
}

const getElements = async (input, IafScriptEngine, context) => {
  let latestModelComposite = await IafScriptEngine.getCompositeCollection({
    query:
    {
      "_userType": "bim_model_version",
      "_namespaces": { "$in": context._namespaces },
      "_itemClass": "NamedCompositeItem"
    }
  }, context, { getLatestVersion: true })
  const latestElementCollection = await IafScriptEngine.getCollectionInComposite(latestModelComposite._id,
    { _userType: "rvt_elements" }, context);
  let bimQuery = {
    "parent": {
      "collectionDesc": {
        "_userType": "rvt_elements",
        "_userItemId": latestElementCollection._userItemId
      },
      "options": {
        "page": {
          "getAllItems": true
        }
      },
      "sort": {
        "_id": 1
      }
    },
    "related": [
      {
        "relatedDesc": {
          "_relatedUserType": "rvt_type_elements"
        },
        "as": "Revit Type Properties"
      },
      {
        "relatedDesc": {
          "_relatedUserType": "rvt_element_props"
        },
        "as": "Revit Element Properties"
      }
    ],
    "relatedFilter": {
      "$and": [
        {
          "relatedDesc": {
            "_relatedUserType": "rvt_type_elements"
          },
          "as": "Revit Type Properties",
          "query": input.params.query
        }
      ]
    }
  };
  let elements = await IafScriptEngine.findWithRelated(bimQuery, context)
  let entities = elements._list.map(e => {
    let properties = {}
    if (!isEmpty(e["Revit Type Properties"]._list[0])) {
      Object.keys(e["Revit Type Properties"]._list[0].properties).forEach((key) => {
        let currentProp = e["Revit Type Properties"]._list[0].properties[key]
        properties[key] = {
          dName: currentProp.dName,
          val: currentProp.val ? currentProp.val : "" + " " + currentProp.uom ? currentProp.uom : "",
          type: "text"
        }
      })
    }
    if (!isEmpty(e["Revit Element Properties"]._list[0])) {
      Object.keys(e["Revit Element Properties"]._list[0].properties).forEach((key) => {
        let currentProp = e["Revit Element Properties"]._list[0].properties[key]
        properties[key] = {
          dName: currentProp.dName,
          val: currentProp.val ? currentProp.val : "" + " " + currentProp.uom ? currentProp.uom : "",
          type: "text"
        }
      })
    }
    let revitFamily = e["Revit Type Properties"]._list[0].properties["Revit Family"] ? e["Revit Type Properties"]._list[0].properties["Revit Family"].val : "No Family"
    let revitType = e["Revit Type Properties"]._list[0].properties["Revit Type"] ? e["Revit Type Properties"]._list[0].properties["Revit Type"].val : "No Type"
    let sysElemId = e["Revit Element Properties"]._list[0].properties.SystemelementId.val
    return {
      _id: e._id,
      "Entity Name": revitFamily + "-" + revitType + "-" + sysElemId,
      properties,
      modelViewerIds: [e.package_id]
    }
  })
  return entities
}

const getRelatedFilesContainer = async (project, P_API, context) => {
  let containers = await P_API.IafFile.getContainers(project, { _name: "Related Files" }, context);
  if (containers === null) {
    let containerInfo = {
      _name: "Related Files",
      _description: "Folder to store related files for model elements.",
      _shortName: "related_files",
      _userType: "file_container",
    };
    let rootContainer = await P_API.IafFile.getRootContainer(project, context);
    if (isEmpty(rootContainer)) {
      rootContainer = await P_API.IafFile.createContainer({ _namespaces: project._namespaces }, undefined, context);
    }
    let newContainer = await P_API.IafFile.createContainer(rootContainer, containerInfo, context);
    return newContainer;
  }
  return containers[0];
}


async function getModelElements(input, libraries, context) {
  const IafScriptEngine = libraries.IafScriptEngine;
  try {
    const P_API = libraries.PlatformApi;
    let project = await P_API.IafProj.getCurrent();
    if (!context) {
      context = { _namespaces: project._namespaces };
    }
    return await getElements(input, IafScriptEngine, context);
  } catch (error) {
    return { error: error.message };
  }
}

async function getRelatedFileItemsForModelElements(input, libraries, context) {
  try {
    const P_API = libraries.PlatformApi;
    let project = await P_API.IafProj.getCurrent();
    if (!context) {
      context = { _namespaces: project._namespaces };
    }
    const container = await getRelatedFilesContainer(project, P_API, context);
    const items = await P_API.IafItemSvc.getRelations(
      container._id,
      {
        query: { _relatedFromId: input.entityInfo._id },
        options: { page: { getAllItems: true } },
      },
      context
    );
    const result = [];
    function formatBytes(bytes, decimals = 2) {
      if (bytes === 0) return "0 Bytes";

      const k = 1024;
      const dm = decimals < 0 ? 0 : decimals;
      const sizes = ["Bytes", "KB", "MB", "GB", "TB", "PB", "EB", "ZB", "YB"];
      const i = Math.floor(Math.log(bytes) / Math.log(k));

      return (
        parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + " " + sizes[i]
      );
    }
    for (let i = 0; i < items._list.length; i++) {
      const file = items._list[i];
      for (let j = 0; j < file._relatedToIds.length; j++) {
        const fileId = file._relatedToIds[j];
        const x = await P_API.IafFile.getFileItem(
          container,
          { _id: fileId },
          context
        );
        const user = await P_API.IafPassSvc.getUserById(
          x._metadata._updatedById,
          context
        );
        result.push({
          name: x.name,
          updatedAt: new Date(x._metadata._updatedAt).toDateString(),
          updatedBy: `${user._firstname} ${user._lastname}`,
          size:
            x.versions[0].fileSize !== undefined
              ? formatBytes(x.versions[0].fileSize)
              : 0,
          fileId: fileId,
          id: x._fileId,
          relatedFromId: input.entityInfo._id,
        });
      }
    }

    if (!result.length) {
      return [{ name: "temp data", is_temp: true }];
    }

    return result;
  } catch (error) {
    return { error: error.message };
  }
}

async function uploadRelatedFileItem(input, libraries, context) {
  console.log("running uploadRelatedFileItem", input);
  try {
    if (
      Array.isArray(input.entityInfo.new) &&
      input.entityInfo.new.length === 0
    ) {
      return;
    }
    const P_API = libraries.PlatformApi;
    let project = await P_API.IafProj.getCurrent();
    if (!context) {
      context = { _namespaces: project._namespaces };
    }
    const configFile = await libraries.UiUtils.IafLocalFile.selectFiles({
      multiple: false,
      accept:
        ".csv, .jpg, .png, .gif, .pdf, .doc, .docx, .rtf, .xls, .xlsx, .ppt, .pptx, .txt",
    });
    if (configFile.length > 0) {
      const container = await getRelatedFilesContainer(
        project,
        P_API,
        context
      );
      const file = await P_API.IafFile.uploadFile(
        container,
        configFile[0].fileObj,
        ["related"],
        context
      );
      let entities = input.entityInfo.new;

      // Ensure entities is always an array
      if (!Array.isArray(entities)) {
        entities = [entities];
      }

      const relationShips = entities.map(entity => ({
        _relatedFromId: entity._id,
        _relatedToIds: [file._id],
        _relatedUserItemDbId: container._id,
      }));
      await P_API.IafItemSvc.addRelations(
        container._id,
        relationShips,
        context
      );

      let actions = document.getElementsByClassName(
        "entity-data-groups-selector"
      )[0]?.childNodes;
      if (actions){
      let childrens = Array.from(actions);
      let fileElem = childrens.find((f) => {
        let item = Array.from(f.children);
        let label = item.find((l) => l.tagName == "LABEL");
        return label.innerText === "Files";
      });
      console.log("🚀 ~ file: Untitled-1:9 ~ fileElem ~ fileElem", fileElem);

      if (fileElem) {
        let fileInput = Array.from(fileElem.children).find(
          (f) => f.tagName == "INPUT"
        );
        if (fileInput.checked) {
          fileInput.click();
          fileInput.click();
        } else {
          fileInput.click();
        }
      }
      }

      // Artifical click of files tab from navigator
      let bottomTabs = document.getElementsByClassName(
        "bottom-panel-content-left"
      )[0]?.childNodes;
      if(bottomTabs){
        let bottomTabsArr = Array.from(bottomTabs);
        let fileTab = bottomTabsArr.find((elem)=> elem.innerText === 'Files');
        fileTab?.click();
      }
    }
  } catch (error) {
    console.log(error);
  }
}

async function deleteRelatedItem(input, libraries, context) {
  try {
    const fileId = input.query.data.fileId;
    const P_API = libraries.PlatformApi;
    let project = await P_API.IafProj.getCurrent();
    if (!context) {
      context = { _namespaces: project._namespaces };
    }
    const container = await getRelatedFilesContainer(project, P_API, context);
    const items = await P_API.IafItemSvc.getRelations(
      container._id,
      {
        query: { _relatedFromId: input.query.data.relatedFromId },
        options: { page: { getAllItems: true } },
      },
      context
    );
    const relation = items._list[0];
    relation._relatedToIds = [fileId];
    await P_API.IafItemSvc.deleteRelations(
      container._id,
      [relation],
      context
    );
  } catch (error) {
    console.log(error);
  }
}

async function getTelemetryCollection(input, libraries, context) {
  try {
    const IafItemSvc = libraries.PlatformApi.IafItemSvc;
    let query = {
      _usertype: "ref_app_telementary_collection",
      _name: "Named Telemetry Collection",
    };
    let itemId = await IafItemSvc.getNamedUserItems(
      query,
      context,
      undefined
    );
    let refAppTelemetryCollection = "";
    for (const ele of itemId._list) {
      if (ele._userType == "ref_app_telementary_collection") {
        refAppTelemetryCollection = ele._id;
      }
    }
    let relatedItem = await IafItemSvc.getRelations(
      refAppTelemetryCollection,
      {
        _relatedFromId: input.entityInfo._id,
      },
      context
    );
    let collectionId = refAppTelemetryCollection;
    let selectedElementId = input.entityInfo._id;
    return { collectionId, selectedElementId };
  } catch (error) {
    return { error: JSON.stringify(error) };
  }
}

async function getIotCollection(input, libraries, context) {
  try {
    const IafItemSvc = libraries.PlatformApi.IafItemSvc;
    let queryIot = {
      _usertype: "ref_app_iot_collection",
      _name: "Named Telemetry Collection",
    };
    let itemIdIot = await IafItemSvc.getNamedUserItems(
      queryIot,
      context,
      undefined
    );
    let refAppIotCollection = "";
    for (const ele of itemIdIot._list) {
      if (ele._userType == "ref_app_iot_collection") {
        refAppIotCollection = ele._id;
      }
    }
    let collectionIdIot = refAppIotCollection;
    let selectedElementId = input.entityInfo._id;
    return { collectionIdIot, selectedElementId };
  } catch (error) {
    return { error: JSON.stringify(error) };
  }
}

async function isValidElement(input, libraries, context) {
  let id = input.modelElementId;
  let { IafScriptEngine } = libraries;
  let latestModelComposite = await IafScriptEngine.getCompositeCollection(
    {
      query: {
        _userType: "bim_model_version",
        _namespaces: { $in: context._namespaces },
        _itemClass: "NamedCompositeItem",
      },
    },
    context,
    { getLatestVersion: true }
  );
  const latestElementCollection =
    await IafScriptEngine.getCollectionInComposite(
      latestModelComposite._id,
      { _userType: "rvt_elements" },
      context
    );
  let bimQuery = {
    parent: {
      collectionDesc: {
        _userType: "rvt_elements",
        _userItemId: latestElementCollection._userItemId,
      },
      query: { _id: `${id}` },
      options: {
        page: {
          getAllItems: true,
        },
      },
      sort: {
        _id: 1,
      },
    },
    related: [
      {
        relatedDesc: {
          _relatedUserType: "rvt_element_props",
        },
        as: "Revit Type Properties",
      },
    ],
  };
  let elements = await IafScriptEngine.findWithRelated(bimQuery, context);
  let valid;
  if (elements._list.length > 0) {
    setResponseProperty('STATUS_CODE', 299)
    const revitTypeProperties = elements._list[0]["Revit Type Properties"];
    const width = revitTypeProperties._list[0].properties["Width"]?.["val"];
    const height = revitTypeProperties._list[0].properties["Height"]?.["val"];
    valid = width > 0 && height > 0;
  } 
  return {
    id,
    isValid: valid,
    latestModelComposite: latestModelComposite,
    latestElementCollection: latestElementCollection,
    elements: elements
  };
}

async function modelElemExists(input, libraries, ctx) {
  const id = input.modelElementId;
  const { IafScriptEngine } = libraries;

  //get latest model composite collection
  const latestModelComposite = await IafScriptEngine.getCompositeCollection(
    {
      query: {
        _userType: "bim_model_version",
        _namespaces: { $in: ctx._namespaces },
        _itemClass: "NamedCompositeItem",
      },
    },
    ctx,
    { getLatestVersion: true }
  );
  //get model elements collection from composite
  const latestElementCollection =
    await IafScriptEngine.getCollectionInComposite(
      latestModelComposite._id,
      { _userType: "rvt_elements" },
      ctx
    );
  console.log("latestElementCollection", latestElementCollection)

  const findInCollQuery = {
    collectionDesc: {
      _userItemId: `${latestElementCollection._userItemId}`
    },
    query: {
      _id: `${id}`
    }
  };

  await IafScriptEngine.findInCollections(findInCollQuery, ctx).then((result) => {
    if (result) {
      if (result.length == 0) {
        setResponseProperty('STATUS_CODE', 498)
        throw new Error(`Element ${id} does not exist in model`);
      } else {
        setResponseProperty('STATUS_CODE', 298)
        return `Element ${id} exists in model`
      }
    } else {
      setResponseProperty('STATUS_CODE', 499)
      throw new Error(`invalid query: ${findInCollQuery}`);
    }
  })
}

async function postSortAndPageFileItemsInColls(input, libraries, ctx) { 
  const { IafItemSvc } = libraries.PlatformApi

  const pageSize = input.params._pageSize || 5;
  const offset = input.params._offset || 0;
  const sortVal = Object.keys(input.params.sort)[0] || "";
  const descending = input.params.sort[sortVal] || 1;
  const query = input.params.query || {};
  let isNumericFlag = false;

  const isNumericSort = ['versions.metadata._updatedAt', 'versions.metadata._createdAt', 'versions.fileSize'];

  const filesQuery = {
    "$findInCollections": {
      "collectionDesc": {
        "_itemClass": "NamedFileCollection",
        "_userType": "file_container"
      },
      "query": {
        ...query
      }
    }
  };
  let matches = [];
  let sortedMatches = [];
  try {
    const queryFileItems = await IafItemSvc.searchRelatedItems(filesQuery, ctx);
    
    queryFileItems._list.map((coll) => {
      if (coll._versions[0]._relatedItems._list.length > 0) {
        coll._versions[0]._relatedItems._list.forEach(item => {
          matches.push(item);
        })
      }
    });

    if (sortVal.length > 0) {
      if (isNumericSort.includes(sortVal)) {
        isNumericFlag = true;
        switch (sortVal) {
          case 'versions.metadata._createdAt': {
            sortedMatches = descending > 0 ?
            matches.slice().sort((a, b) => {
              const aCreatedAt = a.versions[0].metadata._createdAt;
              const bCreatedAt = b.versions[0].metadata._createdAt;
              return aCreatedAt - bCreatedAt;
            })
            : matches.slice().sort((a, b) => {
              const aCreatedAt = a.versions[0].metadata._createdAt;
              const bCreatedAt = b.versions[0].metadata._createdAt;
              return bCreatedAt - aCreatedAt;
            });
            break;
          }
          case 'versions.metadata._updatedAt': {
            sortedMatches = descending > 0 ?
            matches.slice().sort((a, b) => {
              const aCreatedAt = a.versions[0].metadata._updatedAt;
              const bCreatedAt = b.versions[0].metadata._updatedAt;
              return aCreatedAt - bCreatedAt;
            })
            : matches.slice().sort((a, b) => {
              const aCreatedAt = a.versions[0].metadata._updatedAt;
              const bCreatedAt = b.versions[0].metadata._updatedAt;
              return bCreatedAt - aCreatedAt;
            });
            break;
          }
          case 'versions.fileSize': {
            sortedMatches = descending > 0 ?
            matches.slice().sort((a, b) => {
              const aCreatedAt = a.versions[0].fileSize;
              const bCreatedAt = b.versions[0].fileSize;
              return aCreatedAt - bCreatedAt;
            })
            : matches.slice().sort((a, b) => {
              const aCreatedAt = a.versions[0].fileSize;
              const bCreatedAt = b.versions[0].fileSize;
              return bCreatedAt - aCreatedAt;
            });
            break;
          }
          default: 
            sortedMatches = [];
            break;
        }
     
      } else {
        sortedMatches = descending > 0 ?
        matches.slice().sort((a, b) => a[sortVal].localeCompare(b[sortVal]))
        : matches.slice().sort((a, b) => b[sortVal].localeCompare(a[sortVal]));
      }
    }

    return {
      matches: sortedMatches.slice(offset, offset + pageSize),
    }

  } catch (error) {
      throw new Error((error.message))
  }
}