//#region User Config types
let userConfigDescriptors = [
  {
    _name: "DBM Contributor",
    _shortName: "iaf_dbm_contrib_uc",
    _description: "DBM Contributor User Config",
    _userType: "ipa-dt",
  },

  {
    _name: "DBM Viewer",
    _shortName: "iaf_dbm_review_uc",
    _description: "DBM Reviewer User Config",
    _userType: "ipa-dt",
  },

  {
    _name: "DBM Project Admin",
    _shortName: "iaf_dbm_projadmin_uc",
    _description: "DBM Project Admin User Config",
    _userType: "ipa-dt",
  },
];
//#endregion

//*Map between UserConfig and UserGroups; kept separate to avoid using complex projections
let userConfigToUserGroupMap = [
  { userConfig: "iaf_dbm_contrib_uc", userGroup: "file_contrib" },
  { userConfig: "iaf_dbm_review_uc", userGroup: "file_reviewer" },
  { userConfig: "iaf_dbm_projadmin_uc", userGroup: "proj_admin" },
];

//#region Setup User Role Group Configurations
let userGroupDescriptors = [
  {
    _name: "Solutions Mgmt",
    _shortName: "sol_man",
    _description: "Solutions User Group",
    permissions: {
      //accessAll is for easy creation of an admin with access to everything
      accessAll: true,
    },
  },
  {
    _name: "Proj Admin",
    _shortName: "proj_admin",
    _description: "Proj Admin User Group",
    permissions: {
      //accessAll is for easy creation of an admin with access to everything
      accessAll: true,
    },
  },
  {
    _name: "File Contributor",
    _shortName: "file_contrib",
    _description: "File Contributor User Group",
    permissions: {
      workspaces: [{ actions: ["READ", "EDIT"] }],
      namedUserItems: [{ actions: ["READ", "EDIT"] }],
      files: [{ actions: ["READ", "EDIT"] }],
      graphicsdata: [{ actions: ["READ", "CREATE"] }],
      orchestrator: [{ actions: ["READ", "EDIT"] }],
      permissionProfiles: [{ actions: ["READ", "EDIT"] }],
    },
  },
  {
    _name: "Viewer",
    _shortName: "file_reviewer",
    _description: "File Reviewer User Group",
    permissions: {
      workspaces: [{ actions: ["READ"] }],
      namedUserItems: [{ actions: ["READ"] }],
      files: [{ actions: ["READ"] }],
      graphicsdata: [{ actions: ["READ"] }],
      orchestrator: [{ actions: ["READ"] }],
      permissionProfiles: [{ actions: ["READ"] }],
    },
  },
];
//#endregion

const configNames = [
  "iaf_dbm_contrib_uc",
  "iaf_dbm_review_uc",
  "iaf_dbm_projadmin_uc",
];

function Utf8ArrayToStr(array) {
  var out, i, len, c;
  var char2, char3;

  out = "";
  len = array.length;
  i = 0;
  while (i < len) {
    c = array[i++];
    switch (c >> 4) {
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
        // 0xxxxxxx
        out += String.fromCharCode(c);
        break;
      case 12:
      case 13:
        // 110x xxxx   10xx xxxx
        char2 = array[i++];
        out += String.fromCharCode(((c & 0x1f) << 6) | (char2 & 0x3f));
        break;
      case 14:
        // 1110 xxxx  10xx xxxx  10xx xxxx
        char2 = array[i++];
        char3 = array[i++];
        out += String.fromCharCode(
          ((c & 0x0f) << 12) | ((char2 & 0x3f) << 6) | ((char3 & 0x3f) << 0)
        );
        break;
    }
  }

  return out;
}

async function userConifgImports(params, libraries, ctx) {
  console.log("userConifgImports params: ", JSON.stringify(params));
  console.log("userConifgImports ctx: ", JSON.stringify(ctx));
  return new Promise(async (resolve, reject) => {
    try {
      const { PlatformApi, IafScriptEngine, fs } = libraries;
      const { zipLink } = params.actualParams;
      const project = params.actualParams.project._list[0];
      // const ctx = JSON.parse(params.actualParams.ctx);
      const configs = [];

      //* Download project setup zip file
      const response = await fetch(`${zipLink}`);
      const directoryPath = response.unZippedFilePath;
      const filePath = `${directoryPath}/scripts`;
      await readFilesRecursively(filePath);

      async function readFilesRecursively(dir) {
        const files = await fs.readdirSync(dir);
        for (const file of files) {
          const filePath = `${dir}/${file}`;
          if (!file.includes(".")) {
            await readFilesRecursively(filePath); // Await the recursive call
          } else {
            try {
              for (let i = 0; i < configNames.length; i++) {
                const configName = configNames[i];
                if (configName === file.split(".")[0]) {
                  const fileData = await fs.readFileSync(filePath);
                  const dataString = Utf8ArrayToStr(fileData);
                  const dataParsed = JSON.parse(dataString);
                  const configContent = JSON.stringify(dataParsed, null, 2);
                  configs.push({ configName, configContent });
                }
              }
            } catch (error) {
              throw error;
            }
          }
        }
      }

      userGroupDescriptors.forEach((ug) => {
        ug._name = `${project._name} ${ug._name}`
      });

      const userGroups = await PlatformApi.IafProj.addUserGroups(
        project,
        userGroupDescriptors,
        ctx
      );
      console.log("Add User Groups Response.", JSON.stringify(userGroups));

      let configItems = [];
      configs.forEach((c) => {
        let item = userConfigDescriptors.find(
          (obj) => obj._shortName === c.configName
        );
        if (item) {
          item._version = { _userData: c.configContent };
          configItems.push(item);
        }
      });

      //Look up the UserGroup mapped to each UserConfig
      let groupItems = [];
      configs.forEach((c) => {
        let group = userConfigToUserGroupMap.find(
          (obj) => obj.userConfig === c.configName
        );
        let item = userGroups.find((obj) => obj._shortName === group.userGroup);
        if (item) {
          groupItems.push(item);
        }
      });

      for (let i = 0; i < configItems.length; i++) {
        const userConfig = configItems[i];
        const userGroup = groupItems[i];
        let addUserConfigResult =
          await PlatformApi.IafUserGroup.addUserConfigs(
            userGroup,
            [userConfig],
            ctx
          );
        console.log(
          "AddUserConfigs Response:",
          userConfig._name,
          userGroup._name,
          addUserConfigResult
        );
      }
      console.log("Completed userConifgImports", configNames);
      resolve(configNames);
    } catch (err) {
      console.log(err);
      reject(err);
    }
  });
}
