//Definitions for User Scripts to upload
const scriptsDescriptors = [
  {
    _name: "Load Project Collection Data",
    _shortName: "iaf_dt_proj_colls",
    _description: "Load All Project Collections",
    _userType: "iaf_dt_proj_colls",
  },
  {
    _name: "BIMPK Upload",
    _shortName: "iaf_bimpk_upload",
    _description: "Load, Transform and Write Model from BIMPK",
    _userType: "iaf_bimpk_upload",
  },
  {
    _name: "SGPK Upload",
    _shortName: "iaf_sgpk_upload",
    _description: "Load, Transform and Write Model from BIMPK",
    _userType: "iaf_sgpk_upload",
  },
  {
    _name: "BIMPK Post Import - Copy Inverse Relations",
    _shortName: "iaf_bimpk_post_imp",
    _description:
      "BIMPK Post Import - Copy Inverse Relations from Prev Version",
    _userType: "iaf_bimpk_post_imp",
  },
  {
    _name: "Re-mapping type elements",
    _shortName: "iaf_map_elms_type",
    _description: "Update model type elements, after BIMtypes updated",
    _userType: "iaf_map_elms_type",
  },
  {
    _name: "Model Reporting and Validation",
    _shortName: "iaf_ext_val_scr",
    _description: "Scripts to Inspect Model Content and Generate Reports",
    _userType: "iaf_ext_val_scr",
  },
  {
    _name: "Files As Entities All Users",
    _shortName: "iaf_files_allusers",
    _description: "Files for Entity View",
    _userType: "iaf_files_allusers",
  },
  {
    _name: "Files As Entities All Users",
    _shortName: "iaf_dt_model_elems",
    _description: "Model elements",
    _userType: "iaf_dt_model_elems",
  },
  {
    _name: "BMS Equipment Entity scripts",
    _shortName: "iaf_bms_allusers",
    _description: "BMS data as entities",
    _userType: "iaf_bms_allusers",
  },
  {
    _name: "Object Model API Scripts",
    _shortName: "iaf_obj_model_api",
    _description: "Object Model API Asset Script",
    _userType: "iaf_obj_model_api",
  },
  {
    _name: "Import Data Sheets",
    _shortName: "iaf_import_data_sheets",
    _description: "Import Data Sheets",
    _userType: "iaf_import_data_sheets",
  },
  {
    _name: "Orchestrator",
    _shortName: "orchestrator",
    _description: "Schedule orchestrator",
    _userType: "orchestrator",
  },
  {
    _name: "iot_normalization",
    _shortName: "iot_normalization",
    _description: "Script to normalize readings",
    _userType: "telemetry_parser_script",
  },
  {
    _name: "Export Data Sheets",
    _shortName: "iaf_export_data_sheets",
    _description: "Export Data Sheets",
    _userType: "iaf_export_data_sheets",
  },
  {
    _name: "Schemas Scripts",
    _shortName: "iaf_schemas_allusers",
    _description: "Scripts to handle schemas",
    _userType: "iaf_schemas_allusers",
  },
  {
    _name: "Pick List Scripts",
    _shortName: "iaf_pick_lists",
    _description: "Scripts to work with pick lists",
    _userType: "iaf_pick_lists",
  },
  {
    _name: "System Scripts",
    _shortName: "iaf_systems_scripts",
    _description: "Scripts to work with systems",
    _userType: "iaf_systems_scripts",
  },
  {
    _name: "BIMPK Upload Stream",
    _shortName: "iaf_import_model",
    _description: "Load, Transform and Write Model from BIMPK",
    _userType: "iaf_import_model",
  },
  {
    _name: "Files Page Scripts",
    _shortName: "iaf_files_page",
    _description: "Scripts for loading files and folders on the Files Page",
    _userType: "iaf_files_page"
  },
  {
    _name: "Mapbox Scripts",
    _shortName: "iaf_mapbox",
    _description: "Mapbox token request api",
    _userType: "iaf_mapbox"
  }
];

//#region User Config types
let userConfigDescriptors = [
  {
    _name: "DBM Contributor",
    _shortName: "iaf_dbm_contrib_uc",
    _description: "DBM Contributor User Config",
    _userType: "ipa-dt",
  },

  {
    _name: "DBM Viewer",
    _shortName: "iaf_dbm_review_uc",
    _description: "DBM Reviewer User Config",
    _userType: "ipa-dt",
  },

  {
    _name: "DBM Project Admin",
    _shortName: "iaf_dbm_projadmin_uc",
    _description: "DBM Project Admin User Config",
    _userType: "ipa-dt",
  },
];
//#endregion

//*Map between UserConfig and UserGroups; kept separate to avoid using complex projections
let userConfigToUserGroupMap = [
  { userConfig: "iaf_dbm_contrib_uc", userGroup: "file_contrib" },
  { userConfig: "iaf_dbm_review_uc", userGroup: "file_reviewer" },
  { userConfig: "iaf_dbm_projadmin_uc", userGroup: "proj_admin" },
];

//#region Setup User Role Group Configurations
let userGroupDescriptors = [
  {
    _name: "Solutions Mgmt",
    _shortName: "sol_man",
    _description: "Solutions User Group",
    permissions: {
      //accessAll is for easy creation of an admin with access to everything
      accessAll: true,
    },
  },
  {
    _name: "Proj Admin",
    _shortName: "proj_admin",
    _description: "Proj Admin User Group",
    permissions: {
      //accessAll is for easy creation of an admin with access to everything
      accessAll: true,
    },
  },
  {
    _name: "File Contributor",
    _shortName: "file_contrib",
    _description: "File Contributor User Group",
    permissions: {
      workspaces: [{ actions: ["READ", "EDIT"] }],
      namedUserItems: [{ actions: ["READ", "EDIT"] }],
      files: [{ actions: ["READ", "EDIT"] }],
      graphicsdata: [{ actions: ["READ"] }],
    },
  },
  {
    _name: "Viewer",
    _shortName: "file_reviewer",
    _description: "File Reviewer User Group",
    permissions: {
      workspaces: [{ actions: ["READ"] }],
      namedUserItems: [{ actions: ["READ"] }],
      files: [{ actions: ["READ"] }],
      graphicsdata: [{ actions: ["READ"] }],
    },
  },
];
//#endregion

let ProjSetup = {

  async uploadScripts(input, libraries, ctx) {
		const { PlatformApi, UiUtils } = libraries
		const project = await PlatformApi.IafProj.getCurrent(ctx)
		const scriptFiles = await UiUtils.IafLocalFile.selectFiles({ multiple: true, accept: ".js" })
		const scriptNames = _.map(scriptFiles, (f) => {
			return _.split(f.name, '.')[0]
		})
		const scriptContents = await UiUtils.IafLocalFile.loadFiles(scriptFiles)
		const zippedScripts = _.zip(scriptNames, scriptContents)
		const scripts = _.map(zippedScripts, (s) => {
			return { scriptName: s[0], scriptContent: s[1] }
		})

		let scriptItems = []
		scripts.forEach((c) => {
      let item = scriptsDescriptors.find(
        (obj) => obj._shortName === c.scriptName
      );
      if (item) {
        item._version = { _userData: c.scriptContent };
        item._namespaces = project._namespaces;
        scriptItems.push(item);
      }
    });
    console.log("Script items with information.", scriptItems);

		let createScriptRes = await PlatformApi.IafScripts.create(scriptItems, ctx);

		if (createScriptRes && createScriptRes._list) {
			createScriptRes = createScriptRes._list;
		}

		return createScriptRes
  },

  async userConfigImports(input, libraries, ctx) {
		const { PlatformApi, UiUtils } = libraries
		const project = await PlatformApi.IafProj.getCurrent(ctx)
		const configFiles = await UiUtils.IafLocalFile.selectFiles({ multiple: true, accept: ".json" })
		
    const configNames = _.map(configFiles, (f) => {
			//get the first part of the file name without extensions
			return _.split(f.name, '.')[0]
		})
		//load content of the user configs
		const configContents = await UiUtils.IafLocalFile.loadEXPFiles(configFiles)
		const zippedConfigs = _.zip(configNames, configContents)
		const configs = _.map(zippedConfigs, (c) => {
			return { configName: c[0], configContent: c[1] }
		})

    userGroupDescriptors.forEach((ug) => {
      ug._name = `${project._name} ${ug._name}`
    });

    const userGroups = await PlatformApi.IafProj.addUserGroups(
      project,
      userGroupDescriptors,
      ctx
    );
    console.log("Add User Groups Response.", JSON.stringify(userGroups));


		//create configItems
		let configItems = []
    configs.forEach((c) => {
      let item = userConfigDescriptors.find(
        (obj) => obj._shortName === c.configName
      );
      if (item) {
        item._version = { _userData: JSON.stringify(c.configContent, null, 2) };
        configItems.push(item);
      }
    });

		//Look up the UserGroup mapped to each UserConfig
		let groupItems = []
    configs.forEach((c) => {
      let group = userConfigToUserGroupMap.find(
        (obj) => obj.userConfig === c.configName
      );
      let item = userGroups.find((obj) => obj._shortName === group.userGroup);
      if (item) {
        groupItems.push(item);
      }
    });

		let results = []
    for (let i = 0; i < configItems.length; i++) {
      const userConfig = configItems[i];
      const userGroup = groupItems[i];
      let addUserConfigResult =
        await PlatformApi.IafUserGroup.addUserConfigs(
          userGroup,
          [userConfig],
          ctx
        );
      if (addUserConfigResult && addUserConfigResult._list) {
        addUserConfigResult = addUserConfigResult._list;
      }
      results.push(addUserConfigResult);
      console.log(
        "AddUserConfigs Response:",
        userConfig._name,
        userGroup._name,
        addUserConfigResult
      );
    }
    return results
	},

  async bimpkOperations(input, libraries, ctx) {
    //uploadBimpk
		let { PlatformApi, UiUtils, IafScriptEngine } = libraries
		let proj = await PlatformApi.IafProj.getCurrent(ctx);
    console.log("proj at bimpkOperations", proj);
    
		let selectFiles = await UiUtils.IafLocalFile.selectFiles({ 
      multiple: true, accept: ".bimpk, .sgpk"
    })
    const permittedModels = 5;
    if (selectFiles.length > permittedModels) {
      selectFiles = selectFiles.slice(0, permittedModels);
      const selectFileNames = selectFiles.map(f => f.name); 
      console.log(
        `Only 5 models are permitted. Files 1 to 5 will be uploaded: ${selectFileNames}`
      )
    }

    for (let i = 0; i < selectFiles.length; i++) {
      if (selectFiles[i].name.length > 100) {
        throw new Error(
          `The file name ${selectFiles[i].name} 
           contains ${selectFiles[i].name.length} chars. 
           Bimpk file names must be 100 characters or fewer.`
        );
      }
    }

    if (!ctx._namespaces) {
			ctx._namespaces = proj._namespaces
		}
    let uploadedFiles = [];
  
    for (let i = 0; i < selectFiles.length; i++) {
      console.log("running uploadFile")
      //if (selectFiles._name.length > 100) {
        //open dialog
        //dialog has char counter and
        //save button
      //}
      let uploadedFile = await IafScriptEngine.uploadFile(selectFiles[i], ctx);
      uploadedFiles.push(uploadedFile);
      console.log("uploaded bimpk file: ", uploadedFile);  
    }
    console.log("returning uploadFiles: ", uploadedFiles)

    async function addBimpkCollections() {
      const iaf_dt_grid_as_objects = [
        {
          "Revit Category": "OST_BuildingPad",
          "Revit Family": "Pad",
          "Revit Type": "A-M_Pad 1",
        },
        {
          "Revit Category": "OST_CableTray",
          baType: "Cable Trays",
          "Revit Family": "Cable Tray with Fittings",
          "Revit Type": "Submains Cable ladder",
          ElementCategory: "Cable Tray",
          ElementType: "Submains Cable ladder",
        },
        {
          "Revit Category": "OST_CableTray",
          baType: "Cable Trays",
          "Revit Family": "Cable Tray with Fittings",
          "Revit Type": "Channel Cable Tray",
          ElementCategory: "Cable Tray",
          ElementType: "Channel",
        },
      ];
  
      console.log("Started CreateBIMPKDatasource");

      try {
        let atm_defs_coll = await IafScriptEngine.createOrRecreateCollection(
          {
            _name: "ATM Def Collection",
            _shortName: "typemap_defs",
            _namespaces: ctx._namespaces,
            _description: "Asset Type Map Collection",
            _userType: "iaf_dt_type_map_defs_coll",
          },
          ctx
        );
        console.log("ATM Def Collection Create Response.");
  
        let atm_defs_items_res = await IafScriptEngine.createItemsBulk(
          {
            _userItemId: atm_defs_coll._userItemId,
            _namespaces: ctx._namespaces,
            items: iaf_dt_grid_as_objects,
          },
          ctx
        );
        console.log("Create Items Bulk Response.");
      } catch (e) {
        console.log("uploadBimpk addBimpkCollections error", e)
        throw e
      }
    }
    
    console.log("about to run addBimpkCollections");

    await addBimpkCollections();

    return uploadedFiles   
	},

  async importModel(params, libraries, ctx) {
    console.log("running import model");
    const { PlatformApi, IafScriptEngine } = libraries;
  
    try {
  
      // Add importModel datasource (single)
      const importModelDsResult = await IafScriptEngine.addDatasource(
        {
          _name: "BIMPK Import",
          _description: "BIMPK Import",
          _namespaces: ctx._namespaces,
          _userType: "bimpk_import",
          _schemaversion: "2.0",
          _params: {
            tasks: [
              {
                name: "default_script_target",
                _actualparams: {
                  userType: "iaf_import_model",
                  _scriptName: "importModel"
                },
                _sequenceno: 1,
              },
            ],
          },
        },
        ctx
      );
      console.log(
        "Completed importModelDsResult ==== BIMPK Import Add Datasource Response",
        importModelDsResult
      );

      async function fetchBimpkFiles() {
        console.log("SGPK importModels: Fetching BIMPK or SGPK files...");
        const criteria = { 
          _name: ".*(bimpk|sgpk)"
        };
        const fileItems = await PlatformApi.IafFileSvc.getFiles(criteria, ctx);

        if (!fileItems?._list?.length) {
          console.log("SGPK No BIMPK or SGPK files found");
          return [];
        }

        const fileData = await Promise.all(
          fileItems._list.map(async (file) => {
            const fileVersion = await PlatformApi.IafFile.getFileVerisons(file._id, ctx);
            return {
              file_id: file._id,
              fileVersion_id: fileVersion._list[0]._id,
            };
          })
        );

        console.log("SGPK BIMPK files found:", JSON.stringify(fileData));
        return fileData;
      }
          // Fetch BIMPK files
      const bimpkFiles = await fetchBimpkFiles();
      if (!bimpkFiles.length) {
        throw new Error("SGPK No BIMPK files found");
      }
      const bimpkFile = bimpkFiles[0];
      const task = _.find(importModelDsResult.orchsteps, { _sequenceno: 1 });
      const seqTypeId = task._compid;
      const runParam = {
        orchestratorId: importModelDsResult.id,
        _actualparams: [{
          sequence_type_id: seqTypeId,
          params: {
            _fileId: bimpkFile.file_id,
            _fileVersionId: bimpkFile.fileVersion_id
          }
        }]
      }
      // Run the master orchestrator and await its completion with polling
      const mainOrchResult = await IafScriptEngine.runDatasource(
        runParam,
        ctx
      );
      console.log("Master orchestrator started:", mainOrchResult);
  
      // Return a promise that resolves when the main orchestrator completes
      return new Promise((resolve, reject) => {
        let timeoutCounter = 0;
        const maxTimeout = 680; // 30 minutes maximum (180 * 10 seconds)
        
        const interval = setInterval(async () => {
          try {
            timeoutCounter++;
            if (timeoutCounter > maxTimeout) {
              clearInterval(interval);
              reject(new Error("Master orchestrator timed out after 30 minutes"));
              return;
            }
  
            const orchRunResult = await PlatformApi.IafDataSource.getOrchRunStatus(mainOrchResult.id, ctx);
            const orchStepRunStatus = orchRunResult[0].orchrunsteps;
  
            const errStatus = orchStepRunStatus.filter(run_status => run_status._status === "ERROR");
            const queuedStatus = orchStepRunStatus.filter(run_status => run_status._status === "QUEUED");
            const runningStatus = orchStepRunStatus.filter(run_status => run_status._status === "RUNNING");
  
            console.log("Master orchestrator status:", {
              orchRunResult: orchRunResult,
              errors: errStatus.length,
              queued: queuedStatus.length,
              running: runningStatus.length,
              steps: orchStepRunStatus
            });
  
            if (!_.isEmpty(errStatus)) {
              clearInterval(interval);
              reject(new Error(`Master orchestrator encountered errors: ${JSON.stringify(errStatus)}`));
            } else if (_.isEmpty(queuedStatus) && _.isEmpty(runningStatus)) {
              orchStepRunStatus.forEach((step) => step.status = 'COMPLETED');
              clearInterval(interval);
              resolve({
                orchestratorId: mainOrchResult.id,
                status: 'COMPLETED',
                steps: orchStepRunStatus
              });
            }
          } catch (error) {
            clearInterval(interval);
            reject(error);
          }
        }, 10000); // Poll every 10 seconds
      });
    } catch (error) {
      console.error("Error in importModel:", error);
      throw error;
    }
  },

  async createCollections(params, libraries, ctx) {
    const { PlatformApi, UiUtils, IafScriptEngine } = libraries;
    //creates collection search indexes
    let compositeCollections = await IafScriptEngine.getCompositeCollections(
      {
        query: {
          _userType: "bim_model_version",
          _namespaces: {
            $in: ctx._namespaces,
          },
          _itemClass: "NamedCompositeItem",
        },
      },
      ctx,
      { getLatestVersion: true }
    );
    compositeCollections = compositeCollections._list;
    console.log(
      "Started createIndex.",
      "Create index ==== getCompositeCollection.",
      compositeCollections
    );

    const modelNames = compositeCollections.map(m => m._name);

    //user selects files
    const xlsxFiles = await UiUtils.IafLocalFile.selectFiles({ multiple: true, accept: ".xlsx" })

    //check that name format is correct
    for (let i = 0; i < xlsxFiles.length; i++) {
      let missingTypeMap = false;
      let compareName = xlsxFiles[i].name.split('.')[0];
      if (compareName.includes('_TypeMap')) {
        compareName = compareName.replace('_TypeMap', '');
      } else {
        missingTypeMap = true
      }
      if (!modelNames.includes(compareName) || missingTypeMap) {
        throw new Error(
          `The type map file name ${xlsxFiles[i].name} does not match any bimpk file name you uploaded. 
           Your type map file must have the following format: <matching-bimpk-file-name><_TypeMap>.xlsx,
           for example: my-bimpk-filename_TypeMap.xlsx`
        )
      }
    }

    console.log("xlsxFiles.length: ", xlsxFiles.length);
    const typeWorkbooks = await UiUtils.IafDataPlugin.readXLSXFiles(xlsxFiles);
    console.log("typeWorkbooks.length: ", typeWorkbooks.length);
    let returnTypeData = [];
    for (let i = 0; i < xlsxFiles.length; i++) {
      const wbJSON = UiUtils.IafDataPlugin.workbookToJSON(typeWorkbooks[i]);
      const iaf_dt_grid_data = wbJSON.Sheet1;
      const iaf_dt_grid_as_objects = UiUtils.IafDataPlugin.parseGridData({ gridData: iaf_dt_grid_data });
      let typeMapName = xlsxFiles[i].name.split('.')[0];
      if (typeMapName.includes('_TypeMap')) {
        typeMapName = typeMapName.replace('_TypeMap', '');
      }
      const atm_defs_coll = await IafScriptEngine.createOrRecreateCollection(
        {
          _name: typeMapName,
          _shortName: 'typemap_defs',
          _namespaces: ctx._namespaces,
          _description: 'Revit Element Type Map Collection',
          _userType: 'iaf_ref_type_map_defs_coll'
        }, 
        ctx
      )
      console.log("Type Map Collection", atm_defs_coll)
  
      const atm_defs_items_res = await IafScriptEngine.createItemsBulk(
        {
        _userItemId: atm_defs_coll._userItemId,
        _namespaces: ctx._namespaces,
        items: iaf_dt_grid_as_objects
        }, 
        ctx
      );
      console.log("addTypeMap === atm_defs_items_res", atm_defs_items_res);
      returnTypeData.push({
        typeMapCollection: atm_defs_coll,
        typeObjects: atm_defs_items_res
      });
    }

    const createModelIndexes = async (modelCompositeUserItemId) => {
      let collectionInComposite = await IafScriptEngine.getCollectionInComposite(
        modelCompositeUserItemId,
        { _userType: "rvt_type_elements" },
        ctx
      );
      console.log(
        "Create index ==== getCollectionInComposite.",
        collectionInComposite
      );
  
      // Revit
      let indexRes = await IafScriptEngine.createOrRecreateIndex(
        {
          // model type elememt colletcion
          _id: collectionInComposite._userItemId,
          indexDefs: [
            {
              key: { "Element Category": 1 },
              options: {
                name: "model_els_coll_id",
                default_language: "english",
              },
            },
          ],
        },
        ctx
      );
      console.log(
        "Completed createIndex.",
        "Create index ==== Create Or Recreate Index Index response.",
        indexRes
      );
      return indexRes._list[0];
    };

    let indexReses = [];
    for (let i = 0; i < compositeCollections.length; i++) {
      const indexRes = await createModelIndexes(compositeCollections[i]._userItemId);
      console.log("Adding index res: ", indexRes);
      indexReses.push(indexRes);
    }
    console.log("indexReses: ", indexReses);

    const returnData = {
      indexReses: indexReses,
      returnTypeData: returnTypeData
    }

    return returnData;
  },

  async mapRevitTypeCollection(params, libraries, ctx) {
    const { IafScriptEngine, PlatformApi } = libraries;
    try {
      const typeMapMultiDsResult = await IafScriptEngine.addDatasource(
        {
          _name: "Run Element Mapping",
          _description:
            "Orchestrator to map elements to elementCategory and elementType from type map coll",
          _namespaces: ctx._namespaces,
          _userType: "map_type_elems_multi",
          _params: {
            tasks: [
              {
                _orchcomp: "default_script_target",
                _name: "Map type map to elements",
                _sequenceno: 1,
                _actualparams: {
                  userType: "iaf_map_elms_type",
                  _scriptName: "mapRevitTypeCollections",
  
                },
              },
            ],
          },
        },
        ctx
      );
      console.log("addTypeMap === datasourceResult", typeMapMultiDsResult);
      
      const typeMapDatasourceResult = await IafScriptEngine.addDatasource(
        {
          _name: "Map Revit Elements type",
          _description:
            "Orchestrator to map elements to elementCategory and elementType from type map coll",
          _namespaces: ctx._namespaces,
          _userType: "map_revit_type",
          _params: {
            tasks: [
              {
                _orchcomp: "default_script_target",
                _name: "Map type map to elements",
                _sequenceno: 1,
                _actualparams: {
                  userType: "iaf_map_elms_type",
                  _scriptName: "mapRevitTypeCollection",
  
                },
              },
            ],
          },
        },
        ctx
      );
  
      console.log("addTypeMap === datasourceResult", typeMapDatasourceResult);
  
      // Run the master orchestrator and await its completion with polling
      const mainOrchResult = await IafScriptEngine.runDatasource(
        { orchestratorId: typeMapMultiDsResult.id },
        ctx
      );
      console.log("Master orchestrator started:", mainOrchResult);
        
      return new Promise((resolve, reject) => {
        let timeoutCounter = 0;
        const maxTimeout = 680; // 30 minutes maximum (180 * 10 seconds)
        
        const interval = setInterval(async () => {
          try {
            timeoutCounter++;
            if (timeoutCounter > maxTimeout) {
              clearInterval(interval);
              reject(new Error("Master orchestrator timed out after 30 minutes"));
              return;
            }
  
            const orchRunResult = await PlatformApi.IafDataSource.getOrchRunStatus(mainOrchResult.id, ctx);
            const orchStepRunStatus = orchRunResult[0].orchrunsteps;
  
            const errStatus = orchStepRunStatus.filter(run_status => run_status._status === "ERROR");
            const queuedStatus = orchStepRunStatus.filter(run_status => run_status._status === "QUEUED");
            const runningStatus = orchStepRunStatus.filter(run_status => run_status._status === "RUNNING");
  
            console.log("Master orchestrator status:", {
              errors: errStatus.length,
              queued: queuedStatus.length,
              running: runningStatus.length,
              steps: orchStepRunStatus
            });
  
            if (!_.isEmpty(errStatus)) {
              clearInterval(interval);
              reject(new Error(`Master orchestrator encountered errors: ${JSON.stringify(errStatus)}`));
            } else if (_.isEmpty(queuedStatus) && _.isEmpty(runningStatus)) {
              orchStepRunStatus.forEach((step) => step.status = 'COMPLETED');
              clearInterval(interval);
              resolve({
                orchestratorId: mainOrchResult.id,
                status: 'COMPLETED',
                steps: orchStepRunStatus
              });
            }
          } catch (error) {
            clearInterval(interval);
            reject(error);
          }
        }, 10000); // Poll every 10 seconds
      });
    } catch (error) {
      console.error("Error in mapRevitTypeCollection:", error);
      throw error;
    }
  },

  async createOtherCollections(params, libraries, ctx) {
    const { PlatformApi, IafScriptEngine } = libraries;
    const project = await PlatformApi.IafProj.getCurrent(ctx);
    let telemColls = [];
    let iotColls = [];
    let warrantyColls = [];
    let fileColl;
    let permProfile;

    //get bim models
    let bimModels;
    let bimModelsRes = await IafScriptEngine.getCompositeCollections(
      {
        query: {
          _userType: "bim_model_version",
          _namespaces: {
            $in: ctx._namespaces,
          },
          _itemClass: "NamedCompositeItem",
        },
      },
      ctx,
      { getLatestVersion: true }
    );
    console.log("found Models", bimModelsRes);
    bimModels = bimModelsRes._list;

    const createTelemColl = async (model, telemElems, elemColl) => {
      let modelName = model._name;
      if (modelName.includes('.bimpk')) {
        modelName = modelName.replace('.bimpk', '');
      } 
      // Telemetry Collection
      const telemColl =
        await PlatformApi.IafItemSvc.createNamedUserItems(
          [
            {
              _name: `${modelName} Telemetry Collection`,
              _shortName: "telemetry_coll",
              _description:
                "Named Telemetry Collection to store sensor and points data",
              _namespaces: ctx._namespaces,
              _userType: "ref_app_telementary_collection",
              _tsProperties: {
                _granularity: "minutes",
                _expiry: 1440,
              },
            },
          ],
          "NamedTelemetryCollection",
          ctx
        );
      console.log(
        `Started createTelementaryCollection`,
        "createTelementaryCollection ==== Created Telemetry collection response.",
        telemColl
      );
      telemColls.push(telemColl);

      if (telemElems.length > 0) {
        //* Telemetry Collection Relations
        let telemRelatedItems = [
          {
            floorname: "2nd Floor",
            roomname: "Lab",
            _sourceId: "e3e052f9-0156-11d5-9301-0000863f27ad-00000131",
          },
        ];

        let telementryRelatedItems =
          await PlatformApi.IafItemSvc.createRelatedItems(
            telemColl._list[0]._id,
            telemRelatedItems,
            ctx
          );
        console.log(
          "createTelementaryCollection ==== Telementry RelatedItems response.",
          telementryRelatedItems
        );

        let relationShipsSensor1 = [
          {
            _relatedFromId: telemElems[0]._id, // parent id
            _relatedToIds: telementryRelatedItems._list.map((item) => item._id), // children from tip version
            _relatedUserItemDbId: telemColl._list[0]._id,
          },
        ];

        console.log(
          "relationShipsSensor1",
          relationShipsSensor1
        );
        console.log(
          "telemColl._list[0]._id",
          telemColl._list[0]._id
        );
  
        console.log(
          "running telem coll test",
        );
  
        console.log(
          "running relationsSensor1",
        );

        let relationsSensor1 = await PlatformApi.IafItemSvc.addRelations(
          elemColl._itemId,
          relationShipsSensor1,
          ctx
        );
        console.log(
          `Completed createTelementaryCollection`,
          "createTelementaryCollection ==== Telementry add relations response.",
          relationsSensor1
        );
        let relationShipsSensor2 = [
          {
            _relatedFromId: telemElems[1]._id, // parent id
            _relatedToIds: telementryRelatedItems._list.map((item) => item._id), // children from tip version
            _relatedUserItemDbId: telemColl._list[0]._id,
          },
        ];
        console.log(
          "relationShipsSensor2",
          relationShipsSensor2
        );
        // To create relation between pump element id,sensor data and collection
        let relationsSensor2 = await PlatformApi.IafItemSvc.addRelations(
          elemColl._itemId,
          relationShipsSensor2,
          ctx
        );
        console.log(
          `Completed createTelementaryCollection`,
          "createTelementaryCollection ==== Telementry add relations response.",
          relationsSensor2
        );
      }
    };

    const createIotColl = async (model, telemElems, elemColl) => {
      let modelName = model._name;
      if (modelName.includes('.bimpk')) {
        modelName = modelName.replace('.bimpk', '');
      } 
      let IotCollection = await PlatformApi.IafItemSvc.createNamedUserItems(
        [
          {
            _name: `${modelName} Telemetry Collection`,
            _shortName: "telementary_coll",
            _description:
              "Named Telemetry Collection to store sensor and points data",
            _namespaces: ctx._namespaces,
            _userType: "ref_app_iot_collection",
            _tsProperties: {
              _granularity: "minutes",
              _expiry: 1440,
            },
          },
        ],
        "NamedTelemetryCollection",
        ctx
      );
      console.log(
        `Started createIotCollection`,
        "createIotCollection ==== Created Iot collection response.",
        IotCollection
      );
      IotCollection = IotCollection._list[0];
      iotColls.push(IotCollection);
      
      if (telemElems.length > 0) {
        //* Iot Collection Relations
        let relatedItemsIot = [
          {
            floorname: "2nd Floor",
            roomname: "Lab",
            _sourceId: "e3e052f9-0156-11d5-9301-0000863f27ad-00000131",
          },
        ];

        let IotRelatedItems = await PlatformApi.IafItemSvc.createRelatedItems(
          IotCollection._id,
          relatedItemsIot,
          ctx
        );
        console.log(
          "createIotCollection ==== Iot RelatedItems response.",
          IotRelatedItems
        );

        let relationShipsSensorIot = [
          {
            _relatedFromId: telemElems[0]._id, // parent id
            _relatedToIds: IotRelatedItems._list.map((item) => item._id), // children from tip version
            _relatedUserItemDbId: IotCollection._id,
          },
        ];

        let relationsSensorIot = await PlatformApi.IafItemSvc.addRelations(
          elemColl._itemId,
          relationShipsSensorIot,
          ctx
        );
        console.log(
          `Completed createIotCollection`,
          "createIotCollection ==== Iot add relations response.",
          relationsSensorIot
        );
        let relationShipsSensorIot2 = [
          {
            _relatedFromId: telemElems[1]._id, // parent id
            _relatedToIds: IotRelatedItems._list.map((item) => item._id), // children from tip version
            _relatedUserItemDbId: IotCollection._id,
          },
        ];
        // To create relation between pump element id,sensor data and collection
        let relationsSensorIot2 = await PlatformApi.IafItemSvc.addRelations(
          elemColl._itemId,
          relationShipsSensorIot2,
          ctx
        );
        console.log(
          `Completed createIotCollection`,
          "createIotCollection ==== Iot add relations response.",
          relationsSensorIot2
        );
      }
    };

    const createTelemAndIotColls = async (model, elemColl) => { 
      //fetching TelemetryElement 
      let telemElemBimQuery = {
        parent: {
          collectionDesc: {
            _userType: "rvt_elements",
            _userItemId: elemColl._userItemId,
          },
          options: {
            page: {
              getAllItems: true,
            },
          },
          sort: {
            _id: 1,
          },
          query: {
            ElementCategory: { $in: ["Sensor"] },
          },
        },
      };
      let telemElems = await IafScriptEngine.findWithRelated(telemElemBimQuery, ctx);
      console.log("telemElements", telemElems);
      telemElems = telemElems._list

      if (telemElems.length === 0) return;
      
      await createTelemColl(model, telemElems, elemColl);
      await createIotColl(model, telemElems, elemColl);
    }

    const addWarranties = async (model) => {
      // //fetching door elements
      let bimQuery = {
        parent: {
          collectionDesc: {
            _userType: "rvt_elements",
            _userItemId: model._userItemId,
          },
          options: { page: { getAllItems: true } },
          sort: { _id: 1 },
          query: { ElementCategory: { $in: ["Door - External"] } },
        },
      };
      const revit_elements = await IafScriptEngine.findWithRelated(bimQuery, ctx);
      console.log("fetchOSTDoors ==== elements", revit_elements);
        

      //creating warranty collection and relating warranty data
      let modelName = model._name;
      if (modelName.includes('.bimpk')) {
        modelName = modelName.replace('.bimpk', '');
      } 
      let Coll_info = {
        Name: `${model._name} Element Warranty Data`,
        ShortName: "war_data",
        Description: "Element Warranty Data",
        userType: "iaf_dt_warranty_coll",
      };

      let rvt_coll = await IafScriptEngine.getCollectionInComposite(
        model._userItemId,
        { _userType: "rvt_elements" },
        ctx
      );
      console.log("createWarrantyAndElementRelation ==== rvt_coll", rvt_coll);

      let dataObject = {
        properties: {
          "Warranty Description": {
            dName: "Warranty Description",
            val: "The manufacturer warrants this product to be free from defects in workmanship and materials, under normal residential use and conditions, for a period of one (1) year for the original invoice date",
            type: "text",
            epoch: null,
          },
          "Warranty Start Date": {
            dName: "Warranty Start Date",
            val: "09-11-2022",
            type: "date",
          },
          "Warranty Duration": {
            dName: "Warranty Duration",
            val: "1 year",
            type: "text",
          },
        },
      };

      let data_obj_coll = await IafScriptEngine.createOrRecreateCollection(
        {
          _name: Coll_info.Name,
          _shortName: Coll_info.ShortName,
          _namespaces: ctx._namespaces,
          _description: Coll_info.Description,
          _userType: Coll_info.userType,
        },
        ctx
      );
      console.log(
        "createWarrantyAndElementRelation ==== data_obj_coll",
        data_obj_coll
      );

      if (revit_elements._list.length > 0) {
        const flatAllDataObjs = new Array(revit_elements._list.length).fill(
          dataObject
        );
  
        let data_obj_res = await IafScriptEngine.createItemsBulk(
          {
            _userItemId: data_obj_coll._userItemId,
            _namespaces: ctx._namespaces,
            items: flatAllDataObjs,
          },
          ctx
        );
        console.log(
          "createWarrantyAndElementRelation ==== data_obj_res",
          data_obj_res
        );
        warrantyColls.push(data_obj_res);
        //CREATE ITEMS BULK DOESNT RETURN THE CREATED OBJECTS IT RETURNS URIS FOR THE OBJECTS
        //SO GET ALL THOSE IN ONE LIST WHICH WE WILL USE TO GET THE NEW OBJECT IDS OFF THE END
  
        // Get all _uris from data_obj_res[0] and flatten them into a single array
        const allURIs = data_obj_res[0].flatMap((d) => d._uris);
        console.log("createWarrantyAndElementRelation ==== allURIs", allURIs);
  
        // Extract the last part of each URI to get dataObjIds
        const dataObjIds = allURIs.map((u) => u.split("/").pop());
        console.log(
          "createWarrantyAndElementRelation ==== dataObjIds",
          dataObjIds
        );
        // Combine revit_elements and dataObjIds into an array of tuples
        const assetsWithDataIds = revit_elements._list.map((e, i) => [
          e,
          dataObjIds[i],
        ]);
        console.log(
          "createWarrantyAndElementRelation ==== assetsWithDataIds",
          assetsWithDataIds
        );
  
        // Create an array of related items with parentItem and relatedItems properties
        const warrRelatedItems = assetsWithDataIds.map((d) => {
          return { parentItem: d[0], relatedItems: [{ _id: d[1] }] };
        });
        console.log(
          "createWarrantyAndElementRelation ==== warrRelatedItems",
          warrRelatedItems
        );
  
        const relations = await IafScriptEngine.createRelations(
          {
            parentUserItemId: rvt_coll._userItemId,
            _userItemId: data_obj_coll._userItemId,
            _namespaces: ctx._namespaces,
            relations: warrRelatedItems,
          },
          ctx
        );
        console.log("createWarrantyAndElementRelation ==== relations", relations);
  
        //GET AN ASSET WITH THE RELATED DATA AS A TEST
        const testAsset = await IafScriptEngine.findWithRelated(
          {
            parent: {
              query: { _id: revit_elements._list[0]._id },
              collectionDesc: { _userItemId: rvt_coll._userItemId },
              options: { page: { getAllItems: true } },
            },
            related: [
              {
                relatedDesc: { _relatedUserType: data_obj_coll._userType },
                as: "Related Data Object",
              },
            ],
          },
          ctx
        );
        console.log(
          "createWarrantyAndElementRelation ==== testAsset for Warranty Collection : ",
          testAsset._list[0]
        );
        console.log(
          "Completed createWarrantyAndElementRelation.",
          "createWarrantyAndElementRelation ==== testAsset for Warranty Collection : ",
          testAsset._list[0]
        );
      }
    }

    for (let i = 0; i < bimModels.length; i++) {
      try {
        const elemColl =
          await IafScriptEngine.getCollectionInComposite(
            bimModels[i]._id,
            { _userType: "rvt_elements" },
            ctx
          );
        await createTelemAndIotColls(bimModels[i], elemColl);
        await addWarranties(bimModels[i]);
      } catch (e) {
        console.log("addWarranties error", e)
      }
    }

    //creating permission profile
    console.log('Creating permission profile')
    const workspaceId = project._id;
    const irnValue = "passportsvc:workspace:" + workspaceId;
    let viewerPermissions = [
      {
        _name: "viewer_orch_perms1",
        _userType: "viewer_orch_perms",
        _namespaces: [ctx._namespaces[0]],
        _permissions: [
          {
            _actions: ["READ", "SHARE"],
            _namespace: ctx._namespaces[0],
            _resourceDesc: {
              _irn: irnValue,
            },
          },
          {
            _actions: ["READ", "SHARE", "DELETE", "CREATE"],
            _namespace: ctx._namespaces[0],
            _resourceDesc: {
              _irn: "itemsvc:nameduseritem:*",
            },
          },
        ],
      }
    ];
    let result = await PlatformApi.IafPassSvc.createPermissionProfiles(
      viewerPermissions,
      ctx
    );
    console.log("createPermissionProfile ==== result", result);
    permProfile = result;

    try {
      let mapboxPermissions = [
        {
          _name: "mapbox_token_orch_perms",
          _userType: "mapbox_token_perm",
          _namespaces: [ctx._namespaces[0]],
          _permissions: [
            {
              _actions: ["READ", "SHARE"],
              _namespace: ctx._namespaces[0],
              _resourceDesc: {
                _irn: irnValue,
              },
            },
            {
              _actions: ["READ"],
              _namespace: ctx._namespaces[0],
              _resourceDesc: {
                _irn: "itemsvc:nameduseritem:*",
              },
            }
          ]
        }
      ];
      let mapboxResult = await PlatformApi.IafPassSvc.createPermissionProfiles(
        mapboxPermissions,
        ctx
      );
      console.log("createPermissionProfile ==== mapboxResult", mapboxResult);
    } catch (e) {
      console.log("createPermissionProfile ==== mapboxResult error", e);
    }

    //Creating Root Container for files
    console.log("Creating Root Container...");
    const rootContainer = await PlatformApi.IafFile.createContainer(
      project,
      undefined,
      ctx
    );
    console.log("rootContainer result", rootContainer);
    fileColl = rootContainer;

    return {
      telemColls: telemColls,
      iotColls: iotColls,
      warrantyColls: warrantyColls,
      fileColl: fileColl,
      permProfile: permProfile
    }
  },

  async fileAttributesImport(params, libraries, ctx) {
    const { PlatformApi, UiUtils, IafScriptEngine } = libraries;
    const xlsxFiles = await UiUtils.IafLocalFile.selectFiles({ multiple: false, accept: ".xlsx" })
		const proj = await PlatformApi.IafProj.getCurrent(ctx)
    const typeWorkbook = await UiUtils.IafDataPlugin.readXLSXFiles(xlsxFiles)
    const wbJSON = UiUtils.IafDataPlugin.workbookToJSON(typeWorkbook[0]);
    console.log(wbJSON);
    const iaf_dt_grid_data = wbJSON['Document Attributes'];
    console.log("iaf_dt_grid_data", iaf_dt_grid_data);
    let iaf_dt_grid_as_objects = UiUtils.IafDataPlugin.parseGridData(
			{ gridData: iaf_dt_grid_data, options: { asColumns: true } })
    
    console.log("fileAttributesImport ==== iaf_dt_grid_as_objects", iaf_dt_grid_as_objects);

    let file_attr_defs_coll = await IafScriptEngine.createOrRecreateCollection(
      {
        _name: "File Attributes Collection",
        _shortName: "file_attr_defs",
        _namespaces: ctx._namespaces,
        _description: "File Attributes Collection",
        _userType: "iaf_file_attr_defs_coll",
      },
      ctx
    );

    console.log("fileAttributesImport ==== file_attr_defs_coll", file_attr_defs_coll);

    let file_attr_items_res = await IafScriptEngine.createItemsBulk(
      {
        _userItemId: file_attr_defs_coll._userItemId,
        _namespaces: proj._namespaces,
        items: iaf_dt_grid_as_objects,
      },
      ctx
    );
    console.log(
      "fileAttributesImport ==== file_attr_items_res",
      file_attr_items_res
    );
    if (file_attr_defs_coll && file_attr_items_res) {
      return {
        file_attr_defs_coll: file_attr_defs_coll,
        file_attr_items_res: file_attr_items_res
      }      
    }
  },

  async apiConfigImport(input, libraries, ctx) {
    const { PlatformApi, UiUtils } = libraries;
		const configFiles = await UiUtils.IafLocalFile.selectFiles({ multiple: true, accept: ".json" })
    console.log("configFiles", configFiles);
    const configFileData = await UiUtils.IafLocalFile.loadFiles(configFiles);
    const configData = [JSON.parse(configFileData[0])];

    console.log(
      "Started apiConfigImport.",
      "apiConfigImport ==== configData",
      configData
    );

    //Create API configuration with the REST API endpoints
    const configFileItem = await PlatformApi.IafItemSvc.createNamedUserItems(
      [
        {
          _name: "API config",
          _shortName: "api_config",
          _description: "API configuration with the REST API endpoints",
          _namespaces: ctx._namespaces,
          _userType: "api_config",
          _version: { _userData: JSON.stringify(configData) },
        },
      ],
      "UserConfig",
      ctx
    );
    console.log("apiConfigImport ==== configFileItem", configFileItem);

    const configFileItemId = configFileItem?._list?.[0]?._id;
    if (configFileItemId) {
      try {
        const addApiConfigResponse =
          await PlatformApi.IafObjectModelAPISvc.addApiConfig(
            configFileItemId,
            ctx
          );
        console.log(
          "apiConfigImport ==== ApiConfig imported successfully:",
          addApiConfigResponse
        );
        return addApiConfigResponse
      } catch (error) {
        console.error("apiConfigImport ==== Failed to add apiConfig:", error);
      }
    } else {
      console.log("apiConfigImport ==== No configFileItem found");
      return "Failed to add apiConfig"
    }
    console.log("completed apiConfigImport:", configFileItem);
  }
}

export default ProjSetup